'use client'

import { useState } from 'react'

export default function AfricanExercises() {
  const [selectedLevel, setSelectedLevel] = useState('beginner')

  const exercises = {
    beginner: {
      name: 'Beginner',
      description: 'Gentle movements suitable for early pregnancy',
      exercises: [
        {
          id: 1,
          name: 'Afro Dance Flow',
          origin: 'West African tradition',
          duration: '10 mins',
          difficulty: 'Easy',
          description: 'Gentle hip and pelvic movements inspired by traditional Afro dances. These improve pelvic strength and flexibility.',
          steps: [
            'Stand with feet shoulder-width apart',
            'Slowly move hips in figure-8 motion',
            'Let arms flow naturally',
            'Move at your own pace',
            'Rest when needed'
          ],
          benefits: ['Pelvic floor strengthening', 'Improved flexibility', 'Better circulation', 'Stress relief'],
          cautions: 'Stop if you feel dizzy or uncomfortable',
          emoji: '💃🏾'
        },
        {
          id: 2,
          name: 'Walking Meditation (African Style)',
          origin: 'East African wellness practice',
          duration: '15-20 mins',
          difficulty: 'Easy',
          description: 'Mindful walking while connecting with nature, inspired by traditional African practices.',
          steps: [
            'Walk barefoot if possible (on safe ground)',
            'Focus on your breath',
            'Feel connection to earth',
            'Practice gratitude with each step',
            'Walk at natural pace'
          ],
          benefits: ['Cardiovascular health', 'Mental clarity', 'Better balance', 'Pelvic bone preparation'],
          cautions: 'Ensure safe walking surface to prevent falls',
          emoji: '🚶🏾‍♀️'
        },
        {
          id: 3,
          name: 'Pelvic Floor Breathing',
          origin: 'South African doula practices',
          duration: '5-10 mins',
          difficulty: 'Easy',
          description: 'Essential breathing technique to strengthen pelvic floor muscles.',
          steps: [
            'Sit or lie comfortably',
            'Breathe in for 4 counts',
            'Gently engage pelvic floor (like stopping urine)',
            'Hold for 2 counts',
            'Release while breathing out',
            'Repeat 10-15 times'
          ],
          benefits: ['Pelvic floor strength', 'Better bladder control', 'Labor preparation', 'Postpartum recovery'],
          cautions: 'Don\'t overdo if you have pelvic pain',
          emoji: '🫁'
        },
        {
          id: 4,
          name: 'Gentle Cat-Cow Stretches',
          origin: 'African yoga-inspired movements',
          duration: '8 mins',
          difficulty: 'Easy',
          description: 'Inspired by traditional African body awareness practices.',
          steps: [
            'Get on hands and knees',
            'Inhale, drop belly (cow pose)',
            'Exhale, arch back gently (cat pose)',
            'Move slowly with breath',
            'Repeat 10-15 times'
          ],
          benefits: ['Back pain relief', 'Spinal flexibility', 'Improved posture', 'Better breathing'],
          cautions: 'Move gently, no sharp movements',
          emoji: '🐱'
        }
      ]
    },
    intermediate: {
      name: 'Intermediate',
      description: 'Moderate exercises for mid-pregnancy',
      exercises: [
        {
          id: 5,
          name: 'African Dance Cardio',
          origin: 'Zulu and Xhosa dance traditions',
          duration: '20 mins',
          difficulty: 'Moderate',
          description: 'Energetic but controlled African dance movements for cardiovascular fitness.',
          steps: [
            'Start with hip movements',
            'Add shoulder shimmy',
            'Incorporate arm movements',
            'Move to your own rhythm',
            'Rest as needed'
          ],
          benefits: ['Cardiovascular fitness', 'Pelvic mobility', 'Mood enhancement', 'Weight management'],
          cautions: 'Stay hydrated, wear comfortable shoes',
          emoji: '🎶'
        },
        {
          id: 6,
          name: 'Squats (African Harvest Position)',
          origin: 'Traditional harvesting movement',
          duration: '10 mins',
          difficulty: 'Moderate',
          description: 'Inspired by squatting movements used in traditional African agriculture.',
          steps: [
            'Stand with feet shoulder-width apart',
            'Lower into squat position',
            'Keep weight in heels',
            'Hold for 2-3 seconds',
            'Return to standing',
            'Do 12-15 reps'
          ],
          benefits: ['Leg strength', 'Pelvic opening', 'Labor positioning practice', 'Core stability'],
          cautions: 'Support yourself with a chair if needed',
          emoji: '🌾'
        },
        {
          id: 7,
          name: 'Lunges (Market Walk)',
          origin: 'Market trading movement',
          duration: '12 mins',
          difficulty: 'Moderate',
          description: 'Balance exercise inspired by carrying goods in traditional markets.',
          steps: [
            'Step forward with right leg',
            'Lower hips until both knees bent',
            'Push back to starting position',
            'Alternate legs',
            'Do 10-12 reps per leg'
          ],
          benefits: ['Leg and glute strength', 'Balance improvement', 'Hip flexibility', 'Labor preparation'],
          cautions: 'Take your time, use wall for balance if needed',
          emoji: '🏃🏾‍♀️'
        },
        {
          id: 8,
          name: 'Side-Lying Leg Lifts',
          origin: 'Swazi warrior training movements',
          duration: '10 mins',
          difficulty: 'Moderate',
          description: 'Hip and leg strengthening while lying on your side.',
          steps: [
            'Lie on your left side',
            'Keep legs slightly bent',
            'Slowly lift right leg',
            'Lower without touching',
            'Do 15-20 reps',
            'Switch sides'
          ],
          benefits: ['Hip stability', 'Prevent lower back pain', 'Better posture', 'Hip opening'],
          cautions: 'Use pillow between knees for support',
          emoji: '🛏️'
        }
      ]
    },
    advanced: {
      name: 'Advanced',
      description: 'More challenging exercises for later pregnancy (consult doctor first)',
      exercises: [
        {
          id: 9,
          name: 'Birth Ball Exercises',
          origin: 'African wellness traditions adapted',
          duration: '20 mins',
          difficulty: 'Moderate-Advanced',
          description: 'Using birth ball for core strength and labor preparation.',
          steps: [
            'Sit on birth ball with feet flat',
            'Rock gently front to back',
            'Make circular motions',
            'Do gentle bounces',
            'Hold each position 30 seconds'
          ],
          benefits: ['Labor pain relief', 'Pelvic floor strength', 'Core stability', 'Baby positioning'],
          cautions: 'Ensure ball is fully inflated, have chair nearby',
          emoji: '⚪'
        },
        {
          id: 10,
          name: 'Wall Push-Ups (Modified)',
          origin: 'Adapted warrior training',
          duration: '10 mins',
          difficulty: 'Moderate-Advanced',
          description: 'Upper body strength while protecting your belly.',
          steps: [
            'Stand arm\'s length from wall',
            'Place hands on wall',
            'Lean forward with elbows bent',
            'Push back to starting position',
            'Do 10-15 reps'
          ],
          benefits: ['Upper body strength', 'Posture improvement', 'Labor endurance', 'Arm strengthening'],
          cautions: 'Don\'t lean too far forward',
          emoji: '💪🏾'
        },
        {
          id: 11,
          name: 'Bridge Pose (African Meditation)',
          origin: 'South African yoga practices',
          duration: '12 mins',
          difficulty: 'Moderate-Advanced',
          description: 'Glute and lower back strengthening.',
          steps: [
            'Lie on back with knees bent',
            'Feet flat on floor',
            'Slowly lift hips',
            'Hold for 3-5 seconds',
            'Lower gently',
            'Do 12-15 reps'
          ],
          benefits: ['Glute strength', 'Lower back support', 'Core engagement', 'Hip opening'],
          cautions: 'Stop if you feel dizzy or shortness of breath',
          emoji: '🌉'
        },
        {
          id: 12,
          name: 'Kegels (Pelvic Floor Muscle Training)',
          origin: 'Traditional African birth preparation',
          duration: '5 mins daily',
          difficulty: 'Moderate-Advanced',
          description: 'Essential for bladder control and labor readiness.',
          steps: [
            'Identify pelvic floor muscles',
            'Contract them for 3 seconds',
            'Relax for 3 seconds',
            'Repeat 20 times',
            'Rest 1 minute',
            'Do 3 sets daily'
          ],
          benefits: ['Bladder control', 'Better labor outcomes', 'Postpartum recovery', 'Sexual wellness'],
          cautions: 'Can be done anytime, anywhere',
          emoji: '✨'
        }
      ]
    }
  }

  const allExercises = exercises[selectedLevel as 'beginner'|'intermediate'|'advanced'].exercises

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-hotpink to-hotpink-dark rounded-2xl p-8 text-white">
        <h1 className="text-4xl font-bold mb-2">African Prenatal Exercises</h1>
        <p className="text-lg opacity-90">Inspired by traditional African movements for strength & preparation 💪🏾</p>
      </div>

      {/* Level Selector */}
      <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
        <h2 className="text-xl font-bold text-hotpink mb-4">Select Your Fitness Level</h2>
        <div className="grid md:grid-cols-3 gap-4">
          {['beginner', 'intermediate', 'advanced'].map((level) => (
            <button
              key={level}
              onClick={() => setSelectedLevel(level)}
              className={`p-4 rounded-lg font-bold transition text-sm ${
                selectedLevel === level
                  ? 'bg-hotpink text-white'
                  : 'bg-white border-2 border-hotpink-light text-hotpink hover:bg-hotpink/10'
              }`}
            >
              {exercises[level as 'beginner'|'intermediate'|'advanced'].name}
            </button>
          ))}
        </div>
        <p className="text-gray-600 mt-4 text-sm italic">
          {exercises[selectedLevel as 'beginner'|'intermediate'|'advanced'].description}
        </p>
      </div>

      {/* Safety Reminder */}
      <div className="bg-yellow-50 border-2 border-yellow-300 rounded-xl p-6 flex gap-3">
        <span className="text-3xl">⚠️</span>
        <div>
          <p className="font-bold text-yellow-900">Important Safety Information</p>
          <p className="text-sm text-yellow-800 mt-1">
            Always consult your healthcare provider before starting any exercise program. Stop immediately if you experience pain, dizziness, or shortness of breath.
          </p>
        </div>
      </div>

      {/* Exercises */}
      <div className="space-y-4">
        {allExercises.map((exercise) => (
          <div key={exercise.id} className="bg-white rounded-xl border-2 border-hotpink-light p-6 hover:shadow-lg transition">
            <div className="flex items-start justify-between mb-4">
              <div>
                <p className="text-sm font-semibold text-hotpink">From African tradition: {exercise.origin}</p>
                <h3 className="text-2xl font-bold text-deepbrown mt-1">{exercise.emoji} {exercise.name}</h3>
              </div>
              <div className="text-right">
                <div className="bg-hotpink/10 rounded-lg p-2 mb-2">
                  <p className="text-sm font-bold text-hotpink">{exercise.duration}</p>
                </div>
                <div className="bg-terra/10 rounded-lg p-2">
                  <p className="text-sm font-bold text-terra">{exercise.difficulty}</p>
                </div>
              </div>
            </div>

            <p className="text-gray-700 mb-4">{exercise.description}</p>

            {/* Steps */}
            <div className="bg-hotpink/5 rounded-lg p-4 mb-4">
              <p className="font-bold text-deepbrown mb-3">How to Do It:</p>
              <ol className="space-y-2">
                {exercise.steps.map((step, idx) => (
                  <li key={idx} className="flex gap-3 text-gray-700">
                    <span className="font-bold text-hotpink flex-shrink-0">{idx + 1}.</span>
                    <span>{step}</span>
                  </li>
                ))}
              </ol>
            </div>

            {/* Benefits */}
            <div className="grid md:grid-cols-2 gap-4 mb-4">
              <div className="bg-green-50 rounded-lg p-3 border-l-4 border-green-500">
                <p className="font-bold text-green-900 mb-2">Benefits:</p>
                <ul className="text-sm text-green-800 space-y-1">
                  {exercise.benefits.map((benefit, idx) => (
                    <li key={idx}>✓ {benefit}</li>
                  ))}
                </ul>
              </div>

              <div className="bg-orange-50 rounded-lg p-3 border-l-4 border-orange-500">
                <p className="font-bold text-orange-900 mb-2">Cautions:</p>
                <p className="text-sm text-orange-800">{exercise.cautions}</p>
              </div>
            </div>

            <div className="flex gap-3">
              <button className="flex-1 bg-hotpink hover:bg-hotpink-dark text-white font-bold py-2 px-4 rounded-lg transition">
                Start Exercise
              </button>
              <button className="flex-1 border-2 border-hotpink text-hotpink hover:bg-hotpink/10 font-bold py-2 px-4 rounded-lg transition">
                Add to Favorites
              </button>
            </div>
          </div>
        ))}
      </div>

      {/* Exercise Tips */}
      <div className="bg-gradient-to-r from-terra/10 to-terracotta/10 rounded-xl border-l-4 border-terra p-6">
        <h3 className="text-lg font-bold text-deepbrown mb-3 flex items-center gap-2">
          <span>💡</span> African Wisdom on Movement
        </h3>
        <ul className="space-y-2 text-gray-700">
          <li>✓ Movement is medicine - traditional African mothers moved regularly throughout pregnancy</li>
          <li>✓ Connect with your body - pay attention to what feels good</li>
          <li>✓ Dance and rhythm - movement to music aids both body and spirit</li>
          <li>✓ Pelvic strengthening - essential for natural birth readiness</li>
          <li>✓ Listen to your body - rest when needed, modify as necessary</li>
        </ul>
      </div>
    </div>
  )
}
