'use client'

import { useState } from 'react'

interface Milestone {
  id: string
  week: number
  title: string
  notes: string
  date: string
}

interface Measurement {
  id: string
  week: number
  weight: number
  length: number
  headCircumference: number
  date: string
}

export default function EnhancedBabyGrowthTracker() {
  const [activeTab, setActiveTab] = useState('overview')
  const [milestones, setMilestones] = useState<Milestone[]>([
    {
      id: '1',
      week: 0,
      title: 'Birth Day!',
      notes: 'Welcome to the world, little one!',
      date: new Date().toLocaleDateString()
    }
  ])
  const [measurements, setMeasurements] = useState<Measurement[]>([
    {
      id: '1',
      week: 0,
      weight: 3.5,
      length: 50,
      headCircumference: 35,
      date: new Date().toLocaleDateString()
    },
    {
      id: '2',
      week: 4,
      weight: 4.8,
      length: 56,
      headCircumference: 37,
      date: new Date(Date.now() + 28 * 24 * 60 * 60 * 1000).toLocaleDateString()
    }
  ])

  const [newMilestoneTitle, setNewMilestoneTitle] = useState('')
  const [newMilestoneNotes, setNewMilestoneNotes] = useState('')
  const [newMilestoneWeek, setNewMilestoneWeek] = useState(0)

  const [newWeight, setNewWeight] = useState('')
  const [newLength, setNewLength] = useState('')
  const [newHeadCirc, setNewHeadCirc] = useState('')
  const [newMeasWeek, setNewMeasWeek] = useState('')

  const avgWeightByWeek: { [key: number]: number } = {
    0: 3.5, 4: 5.0, 8: 6.0, 12: 7.0, 16: 7.5, 20: 8.0, 24: 8.5,
    28: 9.0, 32: 9.5, 36: 10.0, 40: 10.5, 52: 11.0
  }

  const avgLengthByWeek: { [key: number]: number } = {
    0: 50, 4: 55, 8: 60, 12: 63, 16: 65, 20: 67, 24: 70,
    28: 72, 32: 74, 36: 76, 40: 77, 52: 80
  }

  const developmentMilestones = [
    { week: 0, title: 'Birth', description: 'Baby arrives! Newborn reflexes active', skills: ['Rooting reflex', 'Sucking', 'Grasping'] },
    { week: 4, title: '1 Month', description: 'Baby focusing on faces and high contrast', skills: ['Tracking movement', 'Cooing sounds', 'Social smile'] },
    { week: 8, title: '2 Months', description: 'More alert and responsive', skills: ['Smiling socially', 'Head control improving', 'Following objects'] },
    { week: 12, title: '3 Months', description: 'Baby developing personality', skills: ['Laughing', 'Reaching for objects', 'Rolling attempts'] },
    { week: 16, title: '4 Months', description: 'Increased coordination', skills: ['Rolling over', 'Grasping and shaking toys', 'Better head control'] },
    { week: 20, title: '5 Months', description: 'Greater coordination', skills: ['Sitting with support', 'Transferring objects', 'Making consonant sounds'] },
    { week: 24, title: '6 Months', description: 'Half a year milestone!', skills: ['Sitting without support', 'Rocking back and forth', 'Babbling'] },
    { week: 28, title: '7 Months', description: 'Baby becoming more mobile', skills: ['Crawling', 'Cruising', 'Waving goodbye'] },
    { week: 32, title: '8 Months', description: 'Stranger awareness develops', skills: ['Pulling to stand', 'Pincer grasp', 'Playing peek-a-boo'] },
    { week: 36, title: '9 Months', description: 'Mobile explorers', skills: ['Cruising furniture', 'Clapping', 'Understanding "no"'] },
    { week: 40, title: '10 Months', description: 'Nearly one year!', skills: ['Walking steps', 'Pointing', 'First words'] },
    { week: 52, title: '1 Year', description: 'First birthday!', skills: ['Walking independently', 'Simple words', 'Self-feeding'] }
  ]

  const addMilestone = () => {
    if (newMilestoneTitle) {
      setMilestones([...milestones, {
        id: Date.now().toString(),
        week: newMilestoneWeek,
        title: newMilestoneTitle,
        notes: newMilestoneNotes,
        date: new Date().toLocaleDateString()
      }])
      setNewMilestoneTitle('')
      setNewMilestoneNotes('')
      setNewMilestoneWeek(0)
    }
  }

  const addMeasurement = () => {
    if (newWeight && newLength && newHeadCirc && newMeasWeek) {
      setMeasurements([...measurements, {
        id: Date.now().toString(),
        week: parseInt(newMeasWeek),
        weight: parseFloat(newWeight),
        length: parseFloat(newLength),
        headCircumference: parseFloat(newHeadCirc),
        date: new Date().toLocaleDateString()
      }])
      setNewWeight('')
      setNewLength('')
      setNewHeadCirc('')
      setNewMeasWeek('')
    }
  }

  const latestMeasurement = measurements[measurements.length - 1]
  const currentWeek = latestMeasurement?.week || 0

  const getWeightComparison = (actualWeight: number) => {
    const avg = avgWeightByWeek[currentWeek] || 10
    const diff = actualWeight - avg
    return { diff, percentage: ((diff / avg) * 100).toFixed(1) }
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-hotpink to-hotpink-dark rounded-2xl p-8 text-white">
        <h1 className="text-4xl font-bold mb-2">👶🏾 Baby Growth Tracker</h1>
        <p className="text-lg opacity-90">Track milestones, log measurements, and monitor development progress 📊</p>
      </div>

      {/* Tab Navigation */}
      <div className="flex gap-3 flex-wrap">
        {[
          { id: 'overview', label: '📈 Growth Overview' },
          { id: 'measurements', label: '📏 Measurements' },
          { id: 'milestones', label: '🎯 Milestones' },
          { id: 'development', label: '🧠 Development' }
        ].map(tab => (
          <button
            key={tab.id}
            onClick={() => setActiveTab(tab.id)}
            className={`px-6 py-3 rounded-lg font-bold transition ${
              activeTab === tab.id
                ? 'bg-hotpink text-white shadow-lg'
                : 'bg-white text-hotpink border-2 border-hotpink-light hover:bg-hotpink/10'
            }`}
          >
            {tab.label}
          </button>
        ))}
      </div>

      {/* Growth Overview Tab */}
      {activeTab === 'overview' && (
        <div className="space-y-6">
          {/* Current Stats */}
          <div className="grid md:grid-cols-3 gap-6">
            <div className="bg-gradient-to-br from-hotpink/20 to-hotpink-light/20 rounded-xl p-6 border-2 border-hotpink-light">
              <p className="text-gray-600 text-sm mb-2">Current Week</p>
              <p className="text-4xl font-bold text-hotpink">{currentWeek}w</p>
              <p className="text-xs text-gray-500 mt-2">Age of baby</p>
            </div>

            <div className="bg-gradient-to-br from-gold/20 to-terra/20 rounded-xl p-6 border-2 border-gold">
              <p className="text-gray-600 text-sm mb-2">Weight</p>
              <p className="text-4xl font-bold text-terra">{latestMeasurement?.weight || 0} kg</p>
              {latestMeasurement && (
                <div className="text-xs text-gray-500 mt-2">
                  <p>vs avg</p>
                  <p className={getWeightComparison(latestMeasurement.weight).diff > 0 ? 'text-green-600 font-bold' : 'text-orange-600 font-bold'}>
                    {getWeightComparison(latestMeasurement.weight).percentage}%
                  </p>
                </div>
              )}
            </div>

            <div className="bg-gradient-to-br from-terracotta/20 to-deepbrown/20 rounded-xl p-6 border-2 border-terracotta">
              <p className="text-gray-600 text-sm mb-2">Length</p>
              <p className="text-4xl font-bold text-terracotta">{latestMeasurement?.length || 0} cm</p>
              <p className="text-xs text-gray-500 mt-2">Growing strong!</p>
            </div>
          </div>

          {/* Growth Charts */}
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
            <h2 className="text-2xl font-bold text-hotpink mb-6">Growth Chart Progression</h2>
            
            <div className="space-y-6">
              {measurements.map((m) => (
                <div key={m.id} className="p-4 bg-hotpink/5 rounded-lg">
                  <p className="text-sm font-semibold text-gray-700 mb-3">Week {m.week}: {m.weight}kg, {m.length}cm</p>
                  <div className="space-y-2">
                    <div>
                      <div className="flex gap-2 items-center">
                        <div className="flex-1 bg-gray-200 rounded-full h-4">
                          <div className="bg-hotpink h-4 rounded-full" style={{ width: `${(m.weight / 12) * 100}%` }} />
                        </div>
                        <span className="text-xs text-gray-600">Weight</span>
                      </div>
                    </div>
                    <div>
                      <div className="flex gap-2 items-center">
                        <div className="flex-1 bg-gray-200 rounded-full h-4">
                          <div className="bg-terra h-4 rounded-full" style={{ width: `${(m.length / 85) * 100}%` }} />
                        </div>
                        <span className="text-xs text-gray-600">Length</span>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Measurements Tab */}
      {activeTab === 'measurements' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
            <h2 className="text-2xl font-bold text-hotpink mb-6">Record Measurement</h2>
            
            <div className="grid md:grid-cols-5 gap-4 mb-6">
              <div>
                <label className="block font-bold text-gray-800 mb-2">Week</label>
                <input type="number" value={newMeasWeek} onChange={(e) => setNewMeasWeek(e.target.value)} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
              </div>
              <div>
                <label className="block font-bold text-gray-800 mb-2">Weight (kg)</label>
                <input type="number" step="0.1" value={newWeight} onChange={(e) => setNewWeight(e.target.value)} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
              </div>
              <div>
                <label className="block font-bold text-gray-800 mb-2">Length (cm)</label>
                <input type="number" step="0.1" value={newLength} onChange={(e) => setNewLength(e.target.value)} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
              </div>
              <div>
                <label className="block font-bold text-gray-800 mb-2">Head Circ (cm)</label>
                <input type="number" step="0.1" value={newHeadCirc} onChange={(e) => setNewHeadCirc(e.target.value)} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
              </div>
              <div className="flex items-end">
                <button onClick={addMeasurement} className="w-full py-2 bg-hotpink text-white font-bold rounded-lg hover:bg-hotpink-dark transition">Save</button>
              </div>
            </div>
          </div>

          <div className="bg-white rounded-xl border-2 border-hotpink-light p-8 overflow-x-auto">
            <h2 className="text-2xl font-bold text-hotpink mb-6">Measurement History</h2>
            <table className="w-full text-sm">
              <thead className="border-b-2 border-hotpink-light">
                <tr>
                  <th className="px-4 py-2 text-left font-bold text-hotpink">Week</th>
                  <th className="px-4 py-2 text-left font-bold text-hotpink">Weight</th>
                  <th className="px-4 py-2 text-left font-bold text-hotpink">Length</th>
                  <th className="px-4 py-2 text-left font-bold text-hotpink">Head Circ</th>
                  <th className="px-4 py-2 text-left font-bold text-hotpink">Date</th>
                </tr>
              </thead>
              <tbody>
                {measurements.map((m) => (
                  <tr key={m.id} className="border-b border-hotpink-light/50 hover:bg-hotpink/5">
                    <td className="px-4 py-2">{m.week}w</td>
                    <td className="px-4 py-2">{m.weight} kg</td>
                    <td className="px-4 py-2">{m.length} cm</td>
                    <td className="px-4 py-2">{m.headCircumference} cm</td>
                    <td className="px-4 py-2 text-gray-600">{m.date}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Milestones Tab */}
      {activeTab === 'milestones' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
            <h2 className="text-2xl font-bold text-hotpink mb-6">Log Milestone</h2>
            
            <div className="space-y-4">
              <div className="grid md:grid-cols-2 gap-4">
                <div>
                  <label className="block font-bold text-gray-800 mb-2">Week</label>
                  <input type="number" value={newMilestoneWeek} onChange={(e) => setNewMilestoneWeek(parseInt(e.target.value))} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
                </div>
                <div>
                  <label className="block font-bold text-gray-800 mb-2">Milestone Title</label>
                  <input type="text" value={newMilestoneTitle} onChange={(e) => setNewMilestoneTitle(e.target.value)} placeholder="e.g., First Smile" className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
                </div>
              </div>
              <div>
                <label className="block font-bold text-gray-800 mb-2">Notes</label>
                <textarea value={newMilestoneNotes} onChange={(e) => setNewMilestoneNotes(e.target.value)} rows={3} className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg" />
              </div>
              <button onClick={addMilestone} className="w-full py-3 bg-hotpink text-white font-bold rounded-lg hover:bg-hotpink-dark transition">Save Milestone</button>
            </div>
          </div>

          <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
            <h2 className="text-2xl font-bold text-hotpink mb-6">Milestone Timeline</h2>
            <div className="space-y-4">
              {milestones.sort((a, b) => b.week - a.week).map((m) => (
                <div key={m.id} className="p-4 rounded-lg border-l-4 border-hotpink bg-hotpink/5">
                  <p className="font-bold text-hotpink">{m.title}</p>
                  <p className="text-sm text-gray-600">Week {m.week}</p>
                  <p className="text-gray-700 mt-2">{m.notes}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Development Tab */}
      {activeTab === 'development' && (
        <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
          <h2 className="text-2xl font-bold text-hotpink mb-6">Week-by-Week Development</h2>
          <div className="space-y-6">
            {developmentMilestones.map((dev) => (
              <div key={dev.week} className={`p-6 rounded-lg border-2 ${dev.week === currentWeek ? 'bg-hotpink/10 border-hotpink' : 'bg-white border-hotpink-light'}`}>
                <h3 className="text-xl font-bold text-hotpink mb-2">{dev.title} (Week {dev.week})</h3>
                <p className="text-gray-700 mb-3">{dev.description}</p>
                <p className="font-bold text-gray-800 mb-2">Skills:</p>
                <ul className="space-y-1">
                  {dev.skills.map((skill, i) => (
                    <li key={i} className="text-sm text-gray-600">✓ {skill}</li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
        </div>
      )}

      <div className="bg-terra/20 border-2 border-terra rounded-xl p-6">
        <h3 className="font-bold text-terra mb-3">⚕️ Growth Tips</h3>
        <ul className="space-y-1 text-sm text-gray-700">
          <li>✓ Track measurements every 4 weeks</li>
          <li>✓ Compare to healthy growth ranges</li>
          <li>✓ Celebrate every milestone!</li>
          <li>✓ Contact pediatrician with concerns</li>
        </ul>
      </div>
    </div>
  )
}
