'use client'

import { useState } from 'react'

interface LoggedMeal {
  name: string
  portion: string
  calories: number
  protein: number
  iron: number
  calcium: number
  folate: number
  timestamp: Date
}

export default function FoodIntakeLogger() {
  const [meals, setMeals] = useState<LoggedMeal[]>([])
  const [showForm, setShowForm] = useState(false)
  const [formData, setFormData] = useState({
    name: '',
    portion: '',
    calories: 0,
    protein: 0,
    iron: 0,
    calcium: 0,
    folate: 0
  })

  const commonMeals = [
    { name: 'Ugali with Spinach Stew', calories: 450, protein: 18, iron: 6, calcium: 200, folate: 120 },
    { name: 'Jollof Rice', calories: 380, protein: 8, iron: 3, calcium: 80, folate: 40 },
    { name: 'Lentil Soup', calories: 280, protein: 16, iron: 8, calcium: 150, folate: 180 },
    { name: 'Grilled Fish with Rice', calories: 520, protein: 35, iron: 2, calcium: 100, folate: 50 },
    { name: 'Avocado Toast', calories: 350, protein: 12, iron: 2, calcium: 120, folate: 160 },
    { name: 'Greek Yogurt with Berries', calories: 250, protein: 20, iron: 0.5, calcium: 400, folate: 30 }
  ]

  const addMeal = (meal: typeof commonMeals[0]) => {
    setMeals([
      ...meals,
      {
        ...meal,
        portion: 'Standard',
        timestamp: new Date()
      }
    ])
  }

  const addCustomMeal = () => {
    if (formData.name) {
      setMeals([
        ...meals,
        {
          ...formData,
          timestamp: new Date()
        }
      ])
      setFormData({
        name: '',
        portion: '',
        calories: 0,
        protein: 0,
        iron: 0,
        calcium: 0,
        folate: 0
      })
      setShowForm(false)
    }
  }

  const totalNutrients = meals.reduce(
    (total, meal) => ({
      calories: total.calories + meal.calories,
      protein: total.protein + meal.protein,
      iron: total.iron + meal.iron,
      calcium: total.calcium + meal.calcium,
      folate: total.folate + meal.folate
    }),
    { calories: 0, protein: 0, iron: 0, calcium: 0, folate: 0 }
  )

  return (
    <div className="space-y-6">
      <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
        <h2 className="text-2xl font-bold text-hotpink mb-6">📝 AI-Powered Food Intake Logger</h2>

        {/* Quick Add Buttons */}
        <div className="mb-8">
          <h3 className="font-bold text-gray-800 mb-4">Quick Add Common Meals</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-2">
            {commonMeals.map((meal, idx) => (
              <button
                key={idx}
                onClick={() => addMeal(meal)}
                className="bg-hotpink/10 hover:bg-hotpink/20 text-hotpink font-semibold py-3 px-3 rounded-lg transition text-sm border border-hotpink-light"
              >
                + {meal.name}
              </button>
            ))}
          </div>
        </div>

        {/* Add Custom Meal */}
        <div className="mb-8">
          <button
            onClick={() => setShowForm(!showForm)}
            className="bg-hotpink text-white font-bold py-2 px-6 rounded-lg hover:bg-hotpink-dark transition"
          >
            {showForm ? '✕ Cancel' : '+ Add Custom Meal'}
          </button>

          {showForm && (
            <div className="mt-4 bg-hotpink/5 border-2 border-hotpink-light rounded-lg p-6">
              <input
                type="text"
                placeholder="Meal name"
                value={formData.name}
                onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                className="w-full px-4 py-2 mb-3 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
              />
              
              <div className="grid md:grid-cols-2 gap-3 mb-3">
                <input
                  type="number"
                  placeholder="Calories"
                  value={formData.calories}
                  onChange={(e) => setFormData({ ...formData, calories: Number(e.target.value) })}
                  className="px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
                <input
                  type="number"
                  placeholder="Protein (g)"
                  value={formData.protein}
                  onChange={(e) => setFormData({ ...formData, protein: Number(e.target.value) })}
                  className="px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
                <input
                  type="number"
                  placeholder="Iron (mg)"
                  value={formData.iron}
                  onChange={(e) => setFormData({ ...formData, iron: Number(e.target.value) })}
                  className="px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
                <input
                  type="number"
                  placeholder="Calcium (mg)"
                  value={formData.calcium}
                  onChange={(e) => setFormData({ ...formData, calcium: Number(e.target.value) })}
                  className="px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
                <input
                  type="number"
                  placeholder="Folate (mcg)"
                  value={formData.folate}
                  onChange={(e) => setFormData({ ...formData, folate: Number(e.target.value) })}
                  className="px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
              </div>

              <button
                onClick={addCustomMeal}
                className="w-full bg-hotpink text-white font-bold py-2 rounded-lg hover:bg-hotpink-dark transition"
              >
                Add Meal
              </button>
            </div>
          )}
        </div>

        {/* Logged Meals Today */}
        {meals.length > 0 && (
          <div>
            <h3 className="font-bold text-lg text-hotpink mb-4">Today's Meals ({meals.length})</h3>
            
            {/* Nutrition Summary */}
            <div className="grid md:grid-cols-5 gap-3 mb-6">
              <div className="bg-hotpink/10 rounded-lg p-4 text-center border-2 border-hotpink-light">
                <p className="text-gray-600 text-sm font-semibold">Calories</p>
                <p className="text-2xl font-bold text-hotpink">{Math.round(totalNutrients.calories)}</p>
              </div>
              <div className="bg-gold/10 rounded-lg p-4 text-center border-2 border-gold/50">
                <p className="text-gray-600 text-sm font-semibold">Protein</p>
                <p className="text-2xl font-bold text-gold">{Math.round(totalNutrients.protein)}g</p>
              </div>
              <div className="bg-terra/10 rounded-lg p-4 text-center border-2 border-terra/50">
                <p className="text-gray-600 text-sm font-semibold">Iron</p>
                <p className="text-2xl font-bold text-terra">{totalNutrients.iron.toFixed(1)}mg</p>
              </div>
              <div className="bg-terracotta/10 rounded-lg p-4 text-center border-2 border-terracotta/50">
                <p className="text-gray-600 text-sm font-semibold">Calcium</p>
                <p className="text-2xl font-bold text-terracotta">{Math.round(totalNutrients.calcium)}mg</p>
              </div>
              <div className="bg-deepbrown/10 rounded-lg p-4 text-center border-2 border-deepbrown/50">
                <p className="text-gray-600 text-sm font-semibold">Folate</p>
                <p className="text-2xl font-bold text-deepbrown">{Math.round(totalNutrients.folate)}mcg</p>
              </div>
            </div>

            {/* Meal List */}
            <div className="space-y-2 max-h-96 overflow-y-auto">
              {meals.map((meal, idx) => (
                <div key={idx} className="bg-hotpink/5 border-l-4 border-hotpink rounded-lg p-4 flex justify-between items-center">
                  <div>
                    <p className="font-bold text-gray-800">{meal.name}</p>
                    <div className="flex gap-3 text-xs text-gray-600 mt-1">
                      <span>🔥 {meal.calories} cal</span>
                      <span>🥚 {meal.protein}g protein</span>
                      <span>🩸 {meal.iron}mg iron</span>
                    </div>
                    <p className="text-xs text-gray-500 mt-1">{meal.timestamp.toLocaleTimeString()}</p>
                  </div>
                  <button
                    onClick={() => setMeals(meals.filter((_, i) => i !== idx))}
                    className="text-red-500 hover:text-red-700 font-bold"
                  >
                    ✕
                  </button>
                </div>
              ))}
            </div>
          </div>
        )}

        {meals.length === 0 && (
          <div className="text-center py-12 text-gray-600">
            <p className="text-lg">📋 No meals logged yet</p>
            <p className="text-sm">Start logging meals to track your nutrition!</p>
          </div>
        )}
      </div>

      {/* AI Analysis */}
      <div className="bg-terra/10 border-l-4 border-terra rounded-lg p-6">
        <p className="font-bold text-terra mb-3">🤖 AI Nutrition Analysis</p>
        <ul className="space-y-2 text-sm text-gray-700">
          <li>✓ Your meals are analyzed for pregnancy-specific nutrients</li>
          <li>✓ Get real-time feedback on your nutritional intake</li>
          <li>✓ AI learns your preferences to make better recommendations</li>
          <li>✓ Adjust portions to meet your trimester-specific goals</li>
        </ul>
      </div>
    </div>
  )
}
