'use client'

interface Measurement {
  week: number
  weight: number
  length: number
  headCircumference: number
}

interface GrowthChartsProps {
  measurements: Measurement[]
}

export default function GrowthCharts({ measurements }: GrowthChartsProps) {
  // Average growth standards
  const growthCurves = {
    weight: [
      { week: 0, avg: 3.5, min: 2.5, max: 4.5 },
      { week: 2, avg: 4.2, min: 3.5, max: 5.0 },
      { week: 4, avg: 5.0, min: 4.2, max: 5.8 },
      { week: 8, avg: 6.3, min: 5.5, max: 7.2 },
      { week: 12, avg: 7.5, min: 6.5, max: 8.5 },
      { week: 16, avg: 8.2, min: 7.2, max: 9.2 },
      { week: 20, avg: 9.0, min: 8.0, max: 10.0 },
      { week: 24, avg: 9.8, min: 8.8, max: 10.8 },
      { week: 26, avg: 10.2, min: 9.2, max: 11.2 }
    ],
    length: [
      { week: 0, avg: 50, min: 47, max: 53 },
      { week: 2, avg: 54, min: 51, max: 57 },
      { week: 4, avg: 57, min: 54, max: 60 },
      { week: 8, avg: 62, min: 59, max: 65 },
      { week: 12, avg: 67, min: 64, max: 70 },
      { week: 16, avg: 71, min: 68, max: 74 },
      { week: 20, avg: 74, min: 71, max: 77 },
      { week: 24, avg: 77, min: 74, max: 80 },
      { week: 26, avg: 79, min: 76, max: 82 }
    ],
    head: [
      { week: 0, avg: 35, min: 33, max: 37 },
      { week: 2, avg: 36.5, min: 35, max: 38 },
      { week: 4, avg: 37.8, min: 36.5, max: 39 },
      { week: 8, avg: 39.2, min: 38, max: 40.5 },
      { week: 12, avg: 40.5, min: 39.5, max: 41.5 },
      { week: 16, avg: 41.5, min: 40.5, max: 42.5 },
      { week: 20, avg: 42.3, min: 41.3, max: 43.3 },
      { week: 24, avg: 42.9, min: 41.9, max: 43.9 },
      { week: 26, avg: 43.2, min: 42.2, max: 44.2 }
    ]
  }

  const getMaxValue = (curve: typeof growthCurves.weight) => Math.max(...curve.map(d => d.max)) * 1.1
  const getMinValue = (curve: typeof growthCurves.weight) => Math.min(...curve.map(d => d.min)) * 0.9

  const Chart = ({ title, curve, measurements, unit }: { title: string; curve: Array<{week: number; avg: number; min: number; max: number}>; measurements: Measurement[]; unit: string }) => {
    const maxWeek = Math.max(...measurements.map((m: Measurement) => m.week), ...curve.map(d => d.week))
    const width = Math.min(800, (maxWeek + 2) * 30)
    const chartWidth = width - 60
    const chartHeight = 300

    const maxValue = getMaxValue(curve)
    const minValue = getMinValue(curve)
    const valueRange = maxValue - minValue

    const getX = (week: number) => (week / maxWeek) * chartWidth
    const getY = (value: number) => chartHeight - ((value - minValue) / valueRange) * chartHeight

    const key = title === 'Weight (kg)' ? 'weight' : title === 'Length (cm)' ? 'length' : 'headCircumference'

    return (
      <div className="bg-white border-2 border-hotpink-light rounded-lg p-6 mb-6">
        <h3 className="text-xl font-bold text-hotpink mb-4">{title}</h3>
        
        <svg width={width} height={chartHeight + 50} className="mx-auto">
          {/* Grid Lines */}
          {Array.from({ length: 5 }).map((_, i) => {
            const value = minValue + (i * valueRange) / 4
            const y = getY(value)
            return (
              <g key={`grid-${i}`}>
                <line x1="50" y1={y} x2={chartWidth + 50} y2={y} stroke="#e0e0e0" strokeDasharray="4" />
                <text x="45" y={y + 4} textAnchor="end" fontSize="12" fill="#999">
                  {value.toFixed(1)}
                </text>
              </g>
            )
          })}

          {/* Min/Max Band */}
          <defs>
            <linearGradient id={`gradient-${key}`} x1="0%" y1="0%" x2="0%" y2="100%">
              <stop offset="0%" stopColor="#FFB6D9" stopOpacity="0.3" />
              <stop offset="100%" stopColor="#FFB6D9" stopOpacity="0" />
            </linearGradient>
          </defs>

          {/* Average Line */}
          <polyline
            points={curve.map(d => `${getX(d.week) + 50},${getY(d.avg)}`).join(' ')}
            fill="none"
            stroke="#FF1493"
            strokeWidth="2"
          />

          {/* Min/Max Shading */}
          <polygon
            points={[
              ...curve.map(d => `${getX(d.week) + 50},${getY(d.max)}`),
              ...curve.reverse().map(d => `${getX(d.week) + 50},${getY(d.min)}`)
            ].join(' ')}
            fill={`url(#gradient-${key})`}
            stroke="none"
          />

          {/* Baby's Data Points */}
          {measurements.map((m, idx) => (
            <circle
              key={`point-${idx}`}
              cx={getX(m.week) + 50}
              cy={getY(m[key as keyof Measurement])}
              r="6"
              fill="#FF1493"
              stroke="white"
              strokeWidth="2"
            />
          ))}

          {/* Axes */}
          <line x1="50" y1="0" x2="50" y2={chartHeight} stroke="#000" strokeWidth="2" />
          <line x1="50" y1={chartHeight} x2={chartWidth + 50} y2={chartHeight} stroke="#000" strokeWidth="2" />

          {/* Week Labels */}
          {curve.map((d, i) => (
            <text key={`week-${i}`} x={getX(d.week) + 50} y={chartHeight + 20} textAnchor="middle" fontSize="12">
              W{d.week}
            </text>
          ))}
        </svg>

        {/* Legend */}
        <div className="flex gap-6 mt-6 justify-center">
          <div className="flex items-center gap-2">
            <div className="w-4 h-4 bg-hotpink rounded-full" />
            <span className="text-sm font-semibold">Your Baby</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="h-px w-8 bg-hotpink" />
            <span className="text-sm text-gray-600">Average</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-6 h-4 bg-pink-200 rounded" />
            <span className="text-sm text-gray-600">Normal Range</span>
          </div>
        </div>
      </div>
    )
  }

  return (
    <div className="space-y-6">
      <div className="bg-gradient-to-r from-hotpink/10 to-hotpink-light/10 rounded-xl p-6 border-2 border-hotpink-light">
        <h2 className="text-2xl font-bold text-hotpink mb-4">📊 Growth Charts</h2>
        <p className="text-gray-700">Track your baby's growth with interactive charts comparing to average development curves</p>
      </div>

      {measurements.length === 0 ? (
        <div className="bg-hotpink/10 rounded-lg p-8 text-center border-2 border-dashed border-hotpink-light">
          <p className="text-gray-600">Add measurements to start viewing growth charts 📈</p>
        </div>
      ) : (
        <>
          <Chart title="Weight (kg)" curve={growthCurves.weight} measurements={measurements} unit="kg" />
          <Chart title="Length (cm)" curve={growthCurves.length} measurements={measurements} unit="cm" />
          <Chart title="Head Circumference (cm)" curve={growthCurves.head} measurements={measurements} unit="cm" />
        </>
      )}

      {/* Chart Interpretation */}
      <div className="bg-terra/10 rounded-xl p-8 border-2 border-terra">
        <h3 className="text-2xl font-bold text-terra mb-6">📖 Understanding Growth Charts</h3>
        
        <div className="grid md:grid-cols-3 gap-6">
          <div className="bg-white rounded-lg p-4 border-l-4 border-green-500">
            <p className="font-bold text-green-700 mb-2">✓ On Track</p>
            <p className="text-sm text-gray-700">Baby measurements fall within the normal range (shaded area on chart)</p>
          </div>
          
          <div className="bg-white rounded-lg p-4 border-l-4 border-blue-500">
            <p className="font-bold text-blue-700 mb-2">↑ Above Average</p>
            <p className="text-sm text-gray-700">Baby measurements are slightly above average - still healthy and normal</p>
          </div>
          
          <div className="bg-white rounded-lg p-4 border-l-4 border-yellow-500">
            <p className="font-bold text-yellow-700 mb-2">↓ Below Average</p>
            <p className="text-sm text-gray-700">Baby measurements are slightly below average - discuss with healthcare provider if concerned</p>
          </div>
        </div>

        <div className="mt-6 bg-blue-50 rounded-lg p-4 border-l-4 border-blue-500">
          <p className="font-bold text-blue-900 mb-2">💡 Important Note</p>
          <p className="text-sm text-blue-800">Growth charts show general trends. Every baby grows at their own pace. Regular check-ups with your healthcare provider are the best way to ensure your baby is healthy. Don't worry if measurements vary slightly from averages!</p>
        </div>
      </div>
    </div>
  )
}
