'use client'

import { useState } from 'react'

interface DietaryPreferences {
  dietType: string
  allergies: string[]
  dislikes: string[]
  favorites: string[]
  spiceTolerance: 'none' | 'mild' | 'medium' | 'hot'
}

interface Nutrient {
  calories: number
  protein: number
  iron: number
  calcium: number
  folate: number
}

interface Meal {
  id: string
  name: string
  description: string
  ingredients: string[]
  nutrition: Nutrient
  tags: string[]
  symptoms: string[]
  trimester: string[]
  prepTime: string
  difficulty: 'easy' | 'medium' | 'hard'
  dietCompatible: string[]
  image: string
}

export default function MealAdvisor() {
  const [preferences, setPreferences] = useState<DietaryPreferences>({
    dietType: 'omnivore',
    allergies: [],
    dislikes: [],
    favorites: [],
    spiceTolerance: 'mild'
  })

  const [currentSymptoms, setCurrentSymptoms] = useState<string[]>([])
  const [pregnancyWeek, setPregnancyWeek] = useState(22)
  const [activeTab, setActiveTab] = useState<'advisor' | 'preferences' | 'tracker' | 'logger' | 'calendar'>('advisor')
  const [loggedMeals, setLoggedMeals] = useState<{ meal: Meal; date: string; time: string }[]>([])

  const mealDatabase: Meal[] = [
    // Nausea-friendly meals
    {
      id: 'meal1',
      name: 'Ginger Infused Noodle Soup',
      description: 'Light, warming soup with fresh ginger to combat morning sickness',
      ingredients: ['Rice noodles', 'Fresh ginger', 'Vegetable broth', 'Carrots', 'Spinach', 'Chicken'],
      nutrition: { calories: 320, protein: 18, iron: 2.5, calcium: 120, folate: 85 },
      tags: ['Iron+', 'Folate+', 'Protein+'],
      symptoms: ['nausea', 'fatigue'],
      trimester: ['1', '2', '3'],
      prepTime: '20 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'gluten-free'],
      image: '🍜'
    },
    {
      id: 'meal2',
      name: 'Honey & Banana Toast',
      description: 'Easy-to-digest breakfast with natural energy',
      ingredients: ['Whole grain bread', 'Banana', 'Honey', 'Almond butter'],
      nutrition: { calories: 280, protein: 8, iron: 2, calcium: 100, folate: 45 },
      tags: ['Iron+', 'Energy', 'Quick'],
      symptoms: ['nausea', 'fatigue', 'constipation'],
      trimester: ['1', '2', '3'],
      prepTime: '5 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan'],
      image: '🍯'
    },
    {
      id: 'meal3',
      name: 'Mint & Lemon Water',
      description: 'Refreshing, stomach-soothing hydration',
      ingredients: ['Water', 'Fresh lemon', 'Fresh mint', 'Honey'],
      nutrition: { calories: 15, protein: 0, iron: 0, calcium: 5, folate: 2 },
      tags: ['Hydration', 'Nausea Relief'],
      symptoms: ['nausea', 'heartburn'],
      trimester: ['1', '2', '3'],
      prepTime: '3 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan', 'halal'],
      image: '💧'
    },
    // Iron-rich meals
    {
      id: 'meal4',
      name: 'Spinach & Lentil Curry',
      description: 'Iron-packed African-inspired dish with folate',
      ingredients: ['Red lentils', 'Spinach', 'Coconut milk', 'Turmeric', 'Garlic', 'Onion'],
      nutrition: { calories: 420, protein: 22, iron: 8, calcium: 250, folate: 180 },
      tags: ['Iron+++', 'Folate+++', 'Protein+', 'Calcium+'],
      symptoms: ['fatigue', 'constipation'],
      trimester: ['2', '3'],
      prepTime: '30 min',
      difficulty: 'medium',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan', 'halal'],
      image: '🍛'
    },
    {
      id: 'meal5',
      name: 'Fortified Ugali with Tomato Stew',
      description: 'Traditional African staple with iron-rich greens',
      ingredients: ['Maize flour', 'Leafy greens', 'Tomatoes', 'Onions', 'Red kidney beans'],
      nutrition: { calories: 480, protein: 16, iron: 6, calcium: 180, folate: 120 },
      tags: ['Iron++', 'Calcium+', 'Protein+', 'Folate+'],
      symptoms: ['fatigue', 'anemia'],
      trimester: ['1', '2', '3'],
      prepTime: '35 min',
      difficulty: 'medium',
      dietCompatible: ['omnivore', 'vegetarian', 'halal'],
      image: '🥘'
    },
    // Calcium-rich meals
    {
      id: 'meal6',
      name: 'Greek Yogurt Parfait',
      description: 'Creamy calcium powerhouse with probiotics',
      ingredients: ['Greek yogurt', 'Granola', 'Berries', 'Honey', 'Almonds'],
      nutrition: { calories: 350, protein: 20, iron: 1.5, calcium: 450, folate: 30 },
      tags: ['Calcium+++', 'Protein++'],
      symptoms: ['constipation', 'fatigue'],
      trimester: ['2', '3'],
      prepTime: '5 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian'],
      image: '🥛'
    },
    {
      id: 'meal7',
      name: 'Sesame & Leafy Green Salad',
      description: 'Calcium-rich with African sesame seeds',
      ingredients: ['Sesame seeds', 'Kale', 'Collard greens', 'Beets', 'Olive oil'],
      nutrition: { calories: 280, protein: 14, iron: 4, calcium: 380, folate: 150 },
      tags: ['Calcium++', 'Iron++', 'Folate++'],
      symptoms: ['swelling', 'fatigue'],
      trimester: ['2', '3'],
      prepTime: '15 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan'],
      image: '🥗'
    },
    // Protein-rich meals
    {
      id: 'meal8',
      name: 'Grilled Tilapia with Quinoa',
      description: 'Complete protein with omega-3s',
      ingredients: ['Tilapia', 'Quinoa', 'Lemon', 'Fresh herbs', 'Olive oil'],
      nutrition: { calories: 420, protein: 38, iron: 3, calcium: 140, folate: 95 },
      tags: ['Protein+++', 'Iron+', 'Omega-3'],
      symptoms: ['fatigue', 'muscle cramps'],
      trimester: ['2', '3'],
      prepTime: '25 min',
      difficulty: 'medium',
      dietCompatible: ['omnivore', 'halal'],
      image: '🐟'
    },
    {
      id: 'meal9',
      name: 'Chickpea & Vegetable Stew',
      description: 'Plant-based protein with African spices',
      ingredients: ['Chickpeas', 'Sweet potato', 'Tomatoes', 'Ginger', 'Cumin'],
      nutrition: { calories: 380, protein: 18, iron: 5, calcium: 160, folate: 140 },
      tags: ['Protein++', 'Iron++', 'Folate+'],
      symptoms: ['fatigue', 'constipation'],
      trimester: ['1', '2', '3'],
      prepTime: '40 min',
      difficulty: 'medium',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan', 'halal'],
      image: '🍲'
    },
    // Folate-rich meals
    {
      id: 'meal10',
      name: 'Avocado & Egg Toast',
      description: 'Folate powerhouse for prenatal health',
      ingredients: ['Whole grain bread', 'Avocado', 'Eggs', 'Lemon', 'Hot sauce'],
      nutrition: { calories: 380, protein: 18, iron: 3, calcium: 120, folate: 200 },
      tags: ['Folate+++', 'Protein++', 'Iron+'],
      symptoms: ['fatigue', 'constipation'],
      trimester: ['1', '2', '3'],
      prepTime: '10 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian', 'halal'],
      image: '🥑'
    },
    {
      id: 'meal11',
      name: 'Broccoli Cashew Stir-fry',
      description: 'Green vegetable abundance with healthy fat',
      ingredients: ['Broccoli', 'Cashews', 'Garlic', 'Ginger', 'Brown rice', 'Soy sauce'],
      nutrition: { calories: 420, protein: 16, iron: 4, calcium: 220, folate: 180 },
      tags: ['Folate++', 'Calcium+', 'Iron+'],
      symptoms: ['constipation', 'bloating'],
      trimester: ['1', '2', '3'],
      prepTime: '20 min',
      difficulty: 'easy',
      dietCompatible: ['omnivore', 'vegetarian', 'vegan'],
      image: '🥦'
    }
  ]

  // Pregnancy-specific nutrition goals by trimester
  const nutritionGoals = {
    '1': { calories: 1800, protein: 60, iron: 27, calcium: 1000, folate: 600 },
    '2': { calories: 2200, protein: 70, iron: 27, calcium: 1000, folate: 600 },
    '3': { calories: 2400, protein: 75, iron: 27, calcium: 1000, folate: 600 }
  }

  const getTrimester = (week: number) => {
    if (week <= 13) return '1'
    if (week <= 26) return '2'
    return '3'
  }

  const currentTrimester = getTrimester(pregnancyWeek)

  // Filter meals based on preferences and symptoms
  const recommendedMeals = mealDatabase
    .filter(meal => {
      // Check diet compatibility
      if (!meal.dietCompatible.includes(preferences.dietType)) return false

      // Check allergies
      if (preferences.allergies.some(allergy => meal.ingredients.some(ing => ing.toLowerCase().includes(allergy.toLowerCase())))) {
        return false
      }

      // Check dislikes
      if (preferences.dislikes.some(dislike => meal.ingredients.some(ing => ing.toLowerCase().includes(dislike.toLowerCase())))) {
        return false
      }

      return true
    })
    .sort((a, b) => {
      // Prioritize meals for current symptoms
      const aSymptomMatch = currentSymptoms.filter(s => a.symptoms.includes(s)).length
      const bSymptomMatch = currentSymptoms.filter(s => b.symptoms.includes(s)).length
      if (aSymptomMatch !== bSymptomMatch) return bSymptomMatch - aSymptomMatch

      // Then prioritize for current trimester
      const aTriMatch = a.trimester.includes(currentTrimester) ? 1 : 0
      const bTriMatch = b.trimester.includes(currentTrimester) ? 1 : 0
      return bTriMatch - aTriMatch
    })

  const dailyNutritionTotal = loggedMeals
    .filter(log => log.date === new Date().toISOString().split('T')[0])
    .reduce((total, log) => ({
      calories: total.calories + log.meal.nutrition.calories,
      protein: total.protein + log.meal.nutrition.protein,
      iron: total.iron + log.meal.nutrition.iron,
      calcium: total.calcium + log.meal.nutrition.calcium,
      folate: total.folate + log.meal.nutrition.folate
    }), { calories: 0, protein: 0, iron: 0, calcium: 0, folate: 0 })

  const goals = nutritionGoals[currentTrimester as keyof typeof nutritionGoals]

  const NutritionBar = ({ label, current, goal, unit }: { label: string; current: number; goal: number; unit: string }) => {
    const percentage = Math.min((current / goal) * 100, 100)
    return (
      <div className="mb-4">
        <div className="flex justify-between mb-1">
          <span className="font-semibold text-sm text-gray-700">{label}</span>
          <span className="text-sm text-hotpink font-bold">{Math.round(current)}/{goal} {unit}</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
          <div
            className={`h-full transition-all ${percentage >= 100 ? 'bg-green-500' : 'bg-hotpink'}`}
            style={{ width: `${percentage}%` }}
          />
        </div>
      </div>
    )
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-hotpink to-hotpink-dark rounded-2xl p-8 text-white">
        <h1 className="text-4xl font-bold mb-2">🍽️ AI Meal Advisor</h1>
        <p className="text-lg opacity-90">Personalized pregnancy nutrition with smart recommendations</p>
      </div>

      {/* Tabs */}
      <div className="flex flex-wrap gap-2">
        {[
          { id: 'advisor', label: '🤖 AI Recommendations', icon: '🤖' },
          { id: 'preferences', label: '⚙️ My Preferences', icon: '⚙️' },
          { id: 'tracker', label: '📊 Nutrition Tracker', icon: '📊' },
          { id: 'logger', label: '📝 Log Meals', icon: '📝' },
          { id: 'calendar', label: '📅 Weekly Plan', icon: '📅' }
        ].map((tab) => (
          <button
            key={tab.id}
            onClick={() => setActiveTab(tab.id as any)}
            className={`px-6 py-2 rounded-lg font-bold transition ${
              activeTab === tab.id
                ? 'bg-hotpink text-white shadow-lg'
                : 'bg-white text-hotpink border-2 border-hotpink-light hover:bg-hotpink/10'
            }`}
          >
            {tab.label}
          </button>
        ))}
      </div>

      {/* AI Recommendations Tab */}
      {activeTab === 'advisor' && (
        <div className="space-y-6">
          {/* Symptom Selector */}
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-4">🎯 Today's Symptoms</h2>
            <p className="text-gray-600 mb-4">Select symptoms to get meals that help alleviate them</p>
            
            <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
              {['nausea', 'fatigue', 'constipation', 'heartburn', 'anemia', 'muscle cramps', 'swelling', 'bloating'].map((symptom) => (
                <button
                  key={symptom}
                  onClick={() => {
                    if (currentSymptoms.includes(symptom)) {
                      setCurrentSymptoms(currentSymptoms.filter(s => s !== symptom))
                    } else {
                      setCurrentSymptoms([...currentSymptoms, symptom])
                    }
                  }}
                  className={`px-4 py-2 rounded-lg font-semibold transition capitalize ${
                    currentSymptoms.includes(symptom)
                      ? 'bg-hotpink text-white'
                      : 'bg-hotpink/10 text-hotpink border-2 border-hotpink-light hover:bg-hotpink/20'
                  }`}
                >
                  {symptom}
                </button>
              ))}
            </div>
          </div>

          {/* Recommended Meals */}
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-4">💡 AI-Recommended Meals</h2>
            <p className="text-gray-600 mb-6">Based on week {pregnancyWeek}, {currentSymptoms.length > 0 ? `symptoms (${currentSymptoms.join(', ')})` : 'no active symptoms'}, and your preferences</p>

            <div className="grid md:grid-cols-2 gap-4">
              {recommendedMeals.slice(0, 6).map((meal) => (
                <div key={meal.id} className="border-2 border-hotpink-light rounded-lg p-4 hover:shadow-lg hover:border-hotpink transition">
                  <div className="flex items-start justify-between mb-2">
                    <div>
                      <h3 className="text-xl font-bold text-hotpink">{meal.name}</h3>
                      <p className="text-sm text-gray-600">{meal.description}</p>
                    </div>
                    <span className="text-3xl">{meal.image}</span>
                  </div>

                  <div className="flex gap-2 mb-3 flex-wrap">
                    {meal.tags.map((tag) => (
                      <span key={tag} className="bg-gold/20 text-gold text-xs font-bold px-2 py-1 rounded">
                        {tag}
                      </span>
                    ))}
                  </div>

                  <div className="text-xs text-gray-600 mb-3 space-y-1">
                    <p>⏱️ {meal.prepTime}</p>
                    <p>🔥 {meal.nutrition.calories} cal | 🥚 {meal.nutrition.protein}g protein</p>
                  </div>

                  <button
                    onClick={() => {
                      const now = new Date()
                      setLoggedMeals([...loggedMeals, {
                        meal,
                        date: now.toISOString().split('T')[0],
                        time: now.toLocaleTimeString()
                      }])
                      alert(`${meal.name} logged!`)
                    }}
                    className="w-full bg-hotpink hover:bg-hotpink-dark text-white font-bold py-2 rounded transition"
                  >
                    Log Meal
                  </button>
                </div>
              ))}
            </div>
          </div>

          {/* Why This Meal */}
          <div className="bg-terra/10 border-l-4 border-terra rounded-lg p-6">
            <h3 className="font-bold text-terra mb-2">💪 Why These Meals?</h3>
            <ul className="space-y-2 text-sm text-gray-700">
              <li>✓ Meals ranked by symptom compatibility (top meals address your current symptoms)</li>
              <li>✓ Filtered by your dietary preferences and allergies</li>
              <li>✓ Optimized for Trimester {currentTrimester} nutrition needs</li>
              <li>✓ Rich in pregnancy-essential nutrients (Iron, Calcium, Folate, Protein)</li>
            </ul>
          </div>
        </div>
      )}

      {/* Preferences Tab */}
      {activeTab === 'preferences' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-6">🎯 My Dietary Preferences</h2>

            {/* Diet Type */}
            <div className="mb-8">
              <label className="block font-bold text-gray-800 mb-3">Diet Type</label>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                {['omnivore', 'vegetarian', 'vegan', 'halal'].map((diet) => (
                  <button
                    key={diet}
                    onClick={() => setPreferences({ ...preferences, dietType: diet })}
                    className={`px-4 py-2 rounded-lg font-semibold transition capitalize ${
                      preferences.dietType === diet
                        ? 'bg-hotpink text-white'
                        : 'bg-hotpink/10 text-hotpink border-2 border-hotpink-light'
                    }`}
                  >
                    {diet}
                  </button>
                ))}
              </div>
            </div>

            {/* Spice Tolerance */}
            <div className="mb-8">
              <label className="block font-bold text-gray-800 mb-3">Spice Tolerance</label>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                {['none', 'mild', 'medium', 'hot'].map((level) => (
                  <button
                    key={level}
                    onClick={() => setPreferences({ ...preferences, spiceTolerance: level as any })}
                    className={`px-4 py-2 rounded-lg font-semibold transition capitalize ${
                      preferences.spiceTolerance === level
                        ? 'bg-hotpink text-white'
                        : 'bg-hotpink/10 text-hotpink border-2 border-hotpink-light'
                    }`}
                  >
                    {level}
                  </button>
                ))}
              </div>
            </div>

            {/* Allergies */}
            <div>
              <label className="block font-bold text-gray-800 mb-3">🚫 Allergies</label>
              <div className="flex gap-2 mb-3 flex-wrap">
                {['Peanuts', 'Tree nuts', 'Shellfish', 'Dairy', 'Gluten', 'Eggs'].map((allergy) => (
                  <button
                    key={allergy}
                    onClick={() => {
                      if (preferences.allergies.includes(allergy)) {
                        setPreferences({
                          ...preferences,
                          allergies: preferences.allergies.filter(a => a !== allergy)
                        })
                      } else {
                        setPreferences({
                          ...preferences,
                          allergies: [...preferences.allergies, allergy]
                        })
                      }
                    }}
                    className={`px-3 py-1 rounded-full font-semibold transition text-sm ${
                      preferences.allergies.includes(allergy)
                        ? 'bg-red-500 text-white'
                        : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                    }`}
                  >
                    {allergy}
                  </button>
                ))}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Nutrition Tracker Tab */}
      {activeTab === 'tracker' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-6">📊 Today's Nutrition Intake</h2>
            <p className="text-gray-600 mb-6">Pregnancy Week: <span className="font-bold text-hotpink">{pregnancyWeek}</span> | Trimester: <span className="font-bold text-hotpink">{currentTrimester}</span></p>

            <div className="space-y-6">
              <NutritionBar label="Calories" current={dailyNutritionTotal.calories} goal={goals.calories} unit="kcal" />
              <NutritionBar label="Protein" current={dailyNutritionTotal.protein} goal={goals.protein} unit="g" />
              <NutritionBar label="Iron" current={dailyNutritionTotal.iron} goal={goals.iron} unit="mg" />
              <NutritionBar label="Calcium" current={dailyNutritionTotal.calcium} goal={goals.calcium} unit="mg" />
              <NutritionBar label="Folate" current={dailyNutritionTotal.folate} goal={goals.folate} unit="mcg" />
            </div>

            {loggedMeals.length === 0 && (
              <div className="text-center py-8 text-gray-500">
                <p>No meals logged yet. Start by logging a meal!</p>
              </div>
            )}

            {loggedMeals.length > 0 && (
              <div className="mt-8 pt-8 border-t-2 border-hotpink-light">
                <h3 className="font-bold text-lg text-hotpink mb-4">Today's Meals</h3>
                <div className="space-y-2">
                  {loggedMeals
                    .filter(log => log.date === new Date().toISOString().split('T')[0])
                    .map((log, idx) => (
                      <div key={idx} className="flex justify-between p-3 bg-hotpink/5 rounded-lg">
                        <div>
                          <p className="font-semibold text-gray-800">{log.meal.name}</p>
                          <p className="text-xs text-gray-500">{log.time}</p>
                        </div>
                        <p className="font-bold text-hotpink">{log.meal.nutrition.calories} cal</p>
                      </div>
                    ))}
                </div>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Logger Tab */}
      {activeTab === 'logger' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-6">📝 Log Your Meals</h2>

            <div className="grid md:grid-cols-2 gap-4 mb-6">
              {recommendedMeals.map((meal) => (
                <div key={meal.id} className="border-2 border-hotpink-light rounded-lg p-4 hover:shadow-lg transition">
                  <div className="flex justify-between items-start mb-2">
                    <h3 className="font-bold text-hotpink">{meal.name}</h3>
                    <span className="text-2xl">{meal.image}</span>
                  </div>
                  <p className="text-sm text-gray-600 mb-3">{meal.description}</p>
                  <div className="flex gap-2 mb-3 flex-wrap">
                    {meal.tags.map((tag) => (
                      <span key={tag} className="bg-gold/20 text-gold text-xs font-bold px-2 py-1 rounded">
                        {tag}
                      </span>
                    ))}
                  </div>
                  <button
                    onClick={() => {
                      const now = new Date()
                      setLoggedMeals([...loggedMeals, {
                        meal,
                        date: now.toISOString().split('T')[0],
                        time: now.toLocaleTimeString()
                      }])
                      alert(`${meal.name} logged successfully!`)
                    }}
                    className="w-full bg-hotpink hover:bg-hotpink-dark text-white font-bold py-2 rounded transition"
                  >
                    ✓ Log This Meal
                  </button>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Calendar Tab */}
      {activeTab === 'calendar' && (
        <div className="space-y-6">
          <div className="bg-white rounded-xl border-2 border-hotpink-light p-6">
            <h2 className="text-2xl font-bold text-hotpink mb-6">📅 Weekly Meal Plan</h2>

            <div className="grid md:grid-cols-7 gap-2">
              {['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'].map((day, idx) => (
                <div key={day} className="bg-hotpink/5 border-2 border-hotpink-light rounded-lg p-4">
                  <h3 className="font-bold text-hotpink text-center mb-3">{day}</h3>
                  <div className="space-y-2">
                    {recommendedMeals
                      .slice(idx * 2, (idx + 1) * 2)
                      .map((meal, mealIdx) => (
                        <div key={mealIdx} className="bg-white p-2 rounded border border-hotpink-light text-sm">
                          <p className="font-semibold text-gray-800">{meal.name.split(' ').slice(0, 2).join(' ')}</p>
                          <p className="text-xs text-gray-600">{meal.nutrition.calories} cal</p>
                          <button
                            onClick={() => {
                              setLoggedMeals([...loggedMeals, {
                                meal,
                                date: new Date().toISOString().split('T')[0],
                                time: new Date().toLocaleTimeString()
                              }])
                            }}
                            className="mt-1 w-full bg-hotpink/20 hover:bg-hotpink/40 text-hotpink text-xs font-bold py-1 rounded transition"
                          >
                            Log
                          </button>
                        </div>
                      ))}
                  </div>
                </div>
              ))}
            </div>

            <div className="mt-8 bg-terra/10 border-l-4 border-terra rounded-lg p-4">
              <p className="font-bold text-terra mb-2">💡 Tips:</p>
              <ul className="text-sm text-gray-700 space-y-1">
                <li>✓ Plan your meals weekly to ensure nutritional balance</li>
                <li>✓ Prep meals in advance to manage nausea and fatigue</li>
                <li>✓ Include variety to get all pregnancy nutrients</li>
                <li>✓ Adjust plan based on your symptoms throughout the week</li>
              </ul>
            </div>
          </div>
        </div>
      )}
    </div>
  )
}
