'use client'

import { useState } from 'react'

interface Measurement {
  week: number
  weight: number
  length: number
  headCircumference: number
}

interface MeasurementTrackerProps {
  measurements: Measurement[]
  onAddMeasurement: (measurement: Measurement) => void
}

export default function MeasurementTracker({ measurements, onAddMeasurement }: MeasurementTrackerProps) {
  const [showForm, setShowForm] = useState(false)
  const [formData, setFormData] = useState({
    week: 0,
    weight: '',
    length: '',
    headCircumference: ''
  })

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    
    const newMeasurement: Measurement = {
      week: parseInt(formData.week.toString()),
      weight: parseFloat(formData.weight),
      length: parseFloat(formData.length),
      headCircumference: parseFloat(formData.headCircumference)
    }

    onAddMeasurement(newMeasurement)
    setFormData({
      week: 0,
      weight: '',
      length: '',
      headCircumference: ''
    })
    setShowForm(false)
  }

  // Growth standards for comparison
  const growthStandards: { [key: number]: { weight: number; length: number; head: number } } = {
    0: { weight: 3.5, length: 50, head: 35 },
    2: { weight: 4.5, length: 54, head: 36.5 },
    4: { weight: 5.2, length: 57, head: 37.8 },
    8: { weight: 6.3, length: 62, head: 39.2 },
    12: { weight: 7.5, length: 67, head: 40.5 },
    16: { weight: 8.2, length: 71, head: 41.5 },
    20: { weight: 9.0, length: 74, head: 42.3 },
    24: { weight: 9.8, length: 77, head: 42.9 },
    26: { weight: 10.2, length: 79, head: 43.2 }
  }

  const getGrowthStatus = (babyValue: number, standardValue: number) => {
    const percentage = (babyValue / standardValue) * 100
    if (percentage >= 95 && percentage <= 105) return { status: 'on-track', emoji: '✓' }
    if (percentage > 105) return { status: 'above-average', emoji: '📈' }
    return { status: 'below-average', emoji: '📉' }
  }

  const getStatusColor = (status: string) => {
    if (status === 'on-track') return 'bg-green-100 text-green-900 border-green-300'
    if (status === 'above-average') return 'bg-blue-100 text-blue-900 border-blue-300'
    return 'bg-yellow-100 text-yellow-900 border-yellow-300'
  }

  return (
    <div className="space-y-6">
      {/* Add Measurement Button */}
      <button
        onClick={() => setShowForm(!showForm)}
        className="w-full bg-gradient-to-r from-hotpink to-hotpink-dark text-white font-bold py-3 px-6 rounded-lg hover:shadow-lg transition"
      >
        {showForm ? '✕ Cancel' : '+ Log New Measurement'}
      </button>

      {/* Add Measurement Form */}
      {showForm && (
        <div className="bg-white border-2 border-hotpink-light rounded-xl p-8">
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              <div>
                <label className="block font-bold text-gray-800 mb-2">Baby's Age (Weeks)</label>
                <input
                  type="number"
                  min="0"
                  value={formData.week}
                  onChange={(e) => setFormData({ ...formData, week: parseInt(e.target.value) })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                  required
                />
              </div>

              <div>
                <label className="block font-bold text-gray-800 mb-2">Weight (kg)</label>
                <input
                  type="number"
                  step="0.1"
                  placeholder="e.g., 3.5"
                  value={formData.weight}
                  onChange={(e) => setFormData({ ...formData, weight: e.target.value })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                  required
                />
              </div>
            </div>

            <div className="grid md:grid-cols-2 gap-6">
              <div>
                <label className="block font-bold text-gray-800 mb-2">Length (cm)</label>
                <input
                  type="number"
                  step="0.1"
                  placeholder="e.g., 50"
                  value={formData.length}
                  onChange={(e) => setFormData({ ...formData, length: e.target.value })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                  required
                />
              </div>

              <div>
                <label className="block font-bold text-gray-800 mb-2">Head Circumference (cm)</label>
                <input
                  type="number"
                  step="0.1"
                  placeholder="e.g., 35"
                  value={formData.headCircumference}
                  onChange={(e) => setFormData({ ...formData, headCircumference: e.target.value })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                  required
                />
              </div>
            </div>

            <button
              type="submit"
              className="w-full bg-hotpink text-white font-bold py-3 px-6 rounded-lg hover:bg-hotpink-dark transition"
            >
              Save Measurement
            </button>
          </form>
        </div>
      )}

      {/* Measurements Table */}
      <div className="space-y-4">
        <h2 className="text-2xl font-bold text-hotpink mb-6">📊 Measurement History</h2>
        
        <div className="overflow-x-auto bg-white rounded-xl border-2 border-hotpink-light">
          <table className="w-full">
            <thead className="bg-hotpink text-white">
              <tr>
                <th className="px-6 py-3 text-left font-bold">Week</th>
                <th className="px-6 py-3 text-left font-bold">Weight (kg)</th>
                <th className="px-6 py-3 text-left font-bold">Length (cm)</th>
                <th className="px-6 py-3 text-left font-bold">Head Circumf. (cm)</th>
                <th className="px-6 py-3 text-left font-bold">Status</th>
              </tr>
            </thead>
            <tbody>
              {measurements.map((m, idx) => {
                const standard = growthStandards[m.week] || growthStandards[0]
                const weightStatus = getGrowthStatus(m.weight, standard.weight)
                const lengthStatus = getGrowthStatus(m.length, standard.length)
                const headStatus = getGrowthStatus(m.headCircumference, standard.head)

                return (
                  <tr key={idx} className={idx % 2 === 0 ? 'bg-white' : 'bg-hotpink/5'}>
                    <td className="px-6 py-4 font-semibold text-hotpink">Week {m.week}</td>
                    <td className="px-6 py-4 text-gray-700">{m.weight}</td>
                    <td className="px-6 py-4 text-gray-700">{m.length}</td>
                    <td className="px-6 py-4 text-gray-700">{m.headCircumference}</td>
                    <td className="px-6 py-4">
                      <div className="flex gap-1">
                        <span className={`px-2 py-1 rounded text-xs font-bold border-2 ${getStatusColor(weightStatus.status)}`}>
                          {weightStatus.emoji} {(m.weight / standard.weight * 100).toFixed(0)}%
                        </span>
                      </div>
                    </td>
                  </tr>
                )
              })}
            </tbody>
          </table>
        </div>
      </div>

      {/* Comparison Chart */}
      {measurements.length > 1 && (
        <div className="bg-gradient-to-r from-hotpink/10 to-hotpink-light/10 rounded-xl p-8 border-2 border-hotpink-light">
          <h3 className="text-2xl font-bold text-hotpink mb-6">📈 Growth Comparison</h3>
          
          <div className="grid md:grid-cols-3 gap-6">
            {['Weight (kg)', 'Length (cm)', 'Head Circumference (cm)'].map((label, idx) => {
              const keys = ['weight', 'length', 'headCircumference'] as const
              const key = keys[idx]

              return (
                <div key={idx} className="bg-white rounded-lg p-4 border-2 border-hotpink-light">
                  <p className="font-bold text-gray-800 mb-4">{label}</p>
                  <div className="space-y-2">
                    <div>
                      <div className="flex justify-between text-sm mb-1">
                        <span className="font-semibold text-hotpink">Actual</span>
                        <span>{measurements[measurements.length - 1][key]}</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div
                          className="bg-hotpink h-2 rounded-full"
                          style={{
                            width: '85%'
                          }}
                        />
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-1">
                        <span className="font-semibold text-gray-600">Average</span>
                        <span>{growthStandards[measurements[measurements.length - 1].week]?.[(key === 'headCircumference' ? 'head' : key) as 'weight' | 'length' | 'head'] || 'N/A'}</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-gray-400 h-2 rounded-full w-4/5" />
                      </div>
                    </div>
                  </div>
                </div>
              )
            })}
          </div>
        </div>
      )}

      {/* Growth Scale Info */}
      <div className="bg-terra/10 rounded-xl p-6 border-2 border-terra">
        <h3 className="text-lg font-bold text-terra mb-4">📏 Understanding Growth Measurements</h3>
        
        <div className="grid md:grid-cols-3 gap-4 text-sm">
          <div>
            <p className="font-bold text-terra mb-2">⚖️ Weight</p>
            <p className="text-gray-700">Measured in kilograms (kg). Indicates baby's overall growth and nutrition status.</p>
          </div>
          <div>
            <p className="font-bold text-gold mb-2">📐 Length</p>
            <p className="text-gray-700">Measured in centimeters (cm). Shows bone and overall body growth from head to feet.</p>
          </div>
          <div>
            <p className="font-bold text-deepbrown mb-2">🧠 Head Circumference</p>
            <p className="text-gray-700">Measured in centimeters (cm). Indicates brain growth and development.</p>
          </div>
        </div>
      </div>
    </div>
  )
}
