'use client'

import { useState } from 'react'

interface Milestone {
  id: number
  week: number
  title: string
  description: string
  date: string
  photo: string | null
  weight?: string
  length?: string
}

interface MilestoneLoggerProps {
  milestones: Milestone[]
  onAddMilestone: (milestone: Milestone) => void
}

export default function MilestoneLogger({ milestones, onAddMilestone }: MilestoneLoggerProps) {
  const [showForm, setShowForm] = useState(false)
  const [formData, setFormData] = useState({
    week: 1,
    title: '',
    description: '',
    weight: '',
    length: '',
    photo: null as File | null
  })
  const [photoPreview, setPhotoPreview] = useState<string | null>(null)

  const handlePhotoChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (file) {
      setFormData({ ...formData, photo: file })
      const reader = new FileReader()
      reader.onload = (e) => {
        setPhotoPreview(e.target?.result as string)
      }
      reader.readAsDataURL(file)
    }
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    
    const newMilestone: Milestone = {
      id: Math.max(0, ...milestones.map(m => m.id)) + 1,
      week: formData.week,
      title: formData.title,
      description: formData.description,
      date: new Date().toLocaleDateString(),
      photo: photoPreview,
      weight: formData.weight,
      length: formData.length
    }

    onAddMilestone(newMilestone)
    setFormData({
      week: 1,
      title: '',
      description: '',
      weight: '',
      length: '',
      photo: null
    })
    setPhotoPreview(null)
    setShowForm(false)
  }

  const predefinedMilestones = [
    '🤰 First Scan',
    '💪 First Kick',
    '🎈 Baby Shower',
    '🏥 Hospital Bag Ready',
    '🎉 First Day Home',
    '😊 First Smile',
    '🎂 First Birthday'
  ]

  return (
    <div className="space-y-6">
      {/* Add Milestone Button */}
      <button
        onClick={() => setShowForm(!showForm)}
        className="w-full bg-gradient-to-r from-hotpink to-hotpink-dark text-white font-bold py-3 px-6 rounded-lg hover:shadow-lg transition"
      >
        {showForm ? '✕ Cancel' : '+ Log New Milestone'}
      </button>

      {/* Add Milestone Form */}
      {showForm && (
        <div className="bg-white border-2 border-hotpink-light rounded-xl p-8">
          <form onSubmit={handleSubmit} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-6">
              {/* Week Input */}
              <div>
                <label className="block font-bold text-gray-800 mb-2">Baby's Age (Weeks)</label>
                <input
                  type="number"
                  min="0"
                  value={formData.week}
                  onChange={(e) => setFormData({ ...formData, week: parseInt(e.target.value) })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
              </div>

              {/* Milestone Title */}
              <div>
                <label className="block font-bold text-gray-800 mb-2">Milestone Title</label>
                <input
                  type="text"
                  placeholder="e.g., First Smile"
                  value={formData.title}
                  onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                  required
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
              </div>
            </div>

            {/* Quick Milestone Buttons */}
            <div>
              <label className="block font-bold text-gray-800 mb-2">Quick Milestones</label>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-2">
                {predefinedMilestones.map((milestone) => (
                  <button
                    key={milestone}
                    type="button"
                    onClick={() => setFormData({ ...formData, title: milestone })}
                    className="bg-hotpink/10 text-hotpink border-2 border-hotpink-light hover:bg-hotpink/20 font-semibold py-2 px-3 rounded transition"
                  >
                    {milestone}
                  </button>
                ))}
              </div>
            </div>

            {/* Description */}
            <div>
              <label className="block font-bold text-gray-800 mb-2">Description</label>
              <textarea
                placeholder="Describe this milestone..."
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                rows={3}
              />
            </div>

            {/* Measurements */}
            <div className="grid md:grid-cols-2 gap-6">
              <div>
                <label className="block font-bold text-gray-800 mb-2">Weight (kg)</label>
                <input
                  type="number"
                  step="0.1"
                  placeholder="e.g., 3.5"
                  value={formData.weight}
                  onChange={(e) => setFormData({ ...formData, weight: e.target.value })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
              </div>
              <div>
                <label className="block font-bold text-gray-800 mb-2">Length (cm)</label>
                <input
                  type="number"
                  step="0.1"
                  placeholder="e.g., 50"
                  value={formData.length}
                  onChange={(e) => setFormData({ ...formData, length: e.target.value })}
                  className="w-full px-4 py-2 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
                />
              </div>
            </div>

            {/* Photo Upload */}
            <div>
              <label className="block font-bold text-gray-800 mb-2">📷 Upload Photo</label>
              <div className="border-2 border-dashed border-hotpink-light rounded-lg p-6 text-center hover:bg-hotpink/5 transition cursor-pointer">
                <input
                  type="file"
                  accept="image/*"
                  onChange={handlePhotoChange}
                  className="hidden"
                  id="photo-upload"
                />
                <label htmlFor="photo-upload" className="cursor-pointer">
                  {photoPreview ? (
                    <div className="space-y-2">
                      <img src={photoPreview} alt="Preview" className="w-32 h-32 object-cover mx-auto rounded-lg" />
                      <p className="text-sm text-hotpink font-semibold">Click to change photo</p>
                    </div>
                  ) : (
                    <div className="space-y-2">
                      <p className="text-4xl">📸</p>
                      <p className="text-gray-600">Click to upload a photo of this milestone</p>
                    </div>
                  )}
                </label>
              </div>
            </div>

            {/* Submit Button */}
            <button
              type="submit"
              className="w-full bg-hotpink text-white font-bold py-3 px-6 rounded-lg hover:bg-hotpink-dark transition"
            >
              Save Milestone
            </button>
          </form>
        </div>
      )}

      {/* Milestones Timeline */}
      <div className="space-y-4">
        <h2 className="text-2xl font-bold text-hotpink mb-6">📖 Milestone Timeline</h2>
        
        {milestones.length === 0 ? (
          <div className="bg-hotpink/10 rounded-lg p-8 text-center border-2 border-dashed border-hotpink-light">
            <p className="text-gray-600">No milestones logged yet. Start by adding your first milestone! 🌟</p>
          </div>
        ) : (
          <div className="space-y-4">
            {[...milestones].reverse().map((milestone, idx) => (
              <div key={milestone.id} className="bg-white border-2 border-hotpink-light rounded-lg p-6 hover:shadow-lg transition">
                <div className="flex items-start gap-4">
                  {milestone.photo && (
                    <img src={milestone.photo} alt={milestone.title} className="w-24 h-24 object-cover rounded-lg" />
                  )}
                  
                  <div className="flex-1">
                    <div className="flex items-center justify-between mb-2">
                      <h3 className="text-xl font-bold text-hotpink">{milestone.title}</h3>
                      <span className="text-sm bg-hotpink/20 text-hotpink px-3 py-1 rounded-full font-semibold">
                        Week {milestone.week}
                      </span>
                    </div>
                    
                    <p className="text-sm text-gray-500 mb-2">{milestone.date}</p>
                    <p className="text-gray-700 mb-3">{milestone.description}</p>
                    
                    {(milestone.weight || milestone.length) && (
                      <div className="flex gap-4 text-sm">
                        {milestone.weight && (
                          <span className="bg-gold/20 text-deepbrown px-2 py-1 rounded">
                            ⚖️ {milestone.weight}
                          </span>
                        )}
                        {milestone.length && (
                          <span className="bg-terra/20 text-terra px-2 py-1 rounded">
                            📏 {milestone.length}
                          </span>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  )
}
