'use client'

import { useState } from 'react'

export default function MilestonesEnhancements() {
  const [selectedTrimester, setSelectedTrimester] = useState(0)
  const [expandedFAQ, setExpandedFAQ] = useState<number | null>(null)

  const trimesters = [
    {
      id: 0,
      name: 'First Trimester',
      range: 'Weeks 1-13',
      color: 'bg-red-50',
      borderColor: 'border-red-300',
      focus: 'Foundation Building',
      highlights: [
        'Baby\'s heart starts beating',
        'Brain and nervous system form',
        'Organs begin developing',
        'All major systems established'
      ],
      symptoms: ['Morning sickness', 'Fatigue', 'Mood swings', 'Breast tenderness', 'Frequent urination'],
      selfCare: ['Rest when tired', 'Eat small frequent meals', 'Stay hydrated', 'Take prenatal vitamins'],
      checkups: ['Initial registration', 'Blood tests', 'Ultrasound (8-12 weeks)']
    },
    {
      id: 1,
      name: 'Second Trimester',
      range: 'Weeks 14-26',
      color: 'bg-yellow-50',
      borderColor: 'border-yellow-300',
      focus: 'Growth & Connection',
      highlights: [
        'Baby\'s movements felt',
        'Gender determination possible',
        'Baby\'s facial features visible',
        'Hearing develops'
      ],
      symptoms: ['Improved energy', 'Increased appetite', 'Skin changes', 'Leg cramps'],
      selfCare: ['Regular exercise', 'Prenatal classes', 'Connect with baby', 'Sleep 8-10 hours'],
      checkups: ['Glucose screening', 'Anatomy ultrasound', 'Regular visits']
    },
    {
      id: 2,
      name: 'Third Trimester',
      range: 'Weeks 27-40',
      color: 'bg-purple-50',
      borderColor: 'border-purple-300',
      focus: 'Preparation & Bonding',
      highlights: [
        'Baby gains most weight',
        'Brain develops rapidly',
        'Baby drops into position',
        'Body prepares for labor'
      ],
      symptoms: ['Back pain', 'Swelling', 'Shortness of breath', 'Sleep issues'],
      selfCare: ['Pelvic floor exercises', 'Childbirth classes', 'Relaxation', 'Stay mobile'],
      checkups: ['Weekly visits (near end)', 'Position checks', 'Labor readiness']
    }
  ]

  const pregnancyFAQ = [
    {
      id: 0,
      question: 'Is spotting normal in early pregnancy?',
      answer: 'Light spotting in the first trimester can be normal, but it\'s important to report it to your healthcare provider. They will determine if it\'s implantation bleeding or something requiring attention.'
    },
    {
      id: 1,
      question: 'How much weight should I gain during pregnancy?',
      answer: 'Average weight gain is 11.5-16 kg (25-35 lbs), but this varies based on pre-pregnancy BMI. A healthy weight gain supports fetal development and maternal health. Consult your provider for your personal goal.'
    },
    {
      id: 2,
      question: 'Can I exercise during pregnancy?',
      answer: 'Yes! Regular, moderate exercise is beneficial for most pregnancies. Walking, swimming, prenatal yoga, and strength training are excellent. Avoid contact sports and high-impact activities. Always get clearance from your healthcare provider.'
    },
    {
      id: 3,
      question: 'What foods should I avoid during pregnancy?',
      answer: 'Avoid raw/undercooked meat, unpasteurized dairy, raw eggs, high-mercury fish, and unwashed produce. Limit caffeine to <200mg daily. Eat adequate protein, calcium, and iron-rich foods.'
    },
    {
      id: 4,
      question: 'When do I start feeling baby move?',
      answer: 'First-time mothers usually feel movement around 18-25 weeks. Experienced mothers might feel it earlier (16-18 weeks). Movements feel like flutters, bubbles, or gentle kicks.'
    },
    {
      id: 5,
      question: 'What is gestational diabetes and how is it managed?',
      answer: 'Gestational diabetes occurs during pregnancy and usually resolves after birth. Managed through diet, exercise, and sometimes medication. Your healthcare team will monitor blood sugar levels closely.'
    },
    {
      id: 6,
      question: 'How can I prepare for labor?',
      answer: 'Take childbirth classes, practice relaxation techniques, create a birth plan, learn about pain management options, and ensure your support system is in place. Tour your delivery facility.'
    },
    {
      id: 7,
      question: 'What are Braxton Hicks contractions?',
      answer: 'Practice contractions that prepare your body for labor. They\'re usually painless, irregular, and don\'t increase in intensity. True labor contractions are regular, painful, and progressively closer together.'
    }
  ]

  const weeklyChecklist = [
    {
      trimester: 'First',
      items: [
        'Take prenatal vitamins daily',
        'Eat nutrient-rich foods',
        'Avoid alcohol and smoking',
        'Get adequate rest',
        'Manage stress with relaxation'
      ]
    },
    {
      trimester: 'Second',
      items: [
        'Continue prenatal care visits',
        'Stay active with gentle exercise',
        'Practice relaxation techniques',
        'Attend childbirth classes',
        'Monitor baby movements'
      ]
    },
    {
      trimester: 'Third',
      items: [
        'Prepare nursery and supplies',
        'Practice labor positions',
        'Focus on pelvic floor exercises',
        'Maintain regular checkups',
        'Prepare support system'
      ]
    }
  ]

  const nutritionGuide = [
    {
      nutrient: 'Folic Acid',
      amount: '400-800 mcg daily',
      sources: ['Leafy greens', 'Lentils', 'Fortified cereals', 'Asparagus'],
      importance: 'Prevents neural tube defects'
    },
    {
      nutrient: 'Iron',
      amount: '27 mg daily',
      sources: ['Red meat', 'Chicken', 'Fish', 'Beans', 'Leafy greens'],
      importance: 'Prevents anemia, supports oxygen transport'
    },
    {
      nutrient: 'Calcium',
      amount: '1000 mg daily',
      sources: ['Dairy', 'Leafy greens', 'Almonds', 'Sesame seeds'],
      importance: 'Supports baby\'s bone development'
    },
    {
      nutrient: 'Protein',
      amount: '70-100g daily',
      sources: ['Eggs', 'Fish', 'Legumes', 'Nuts', 'Seeds'],
      importance: 'Essential for fetal growth and development'
    }
  ]

  const birthPlanTemplate = [
    { category: 'Environment', options: ['Dim lighting', 'Music', 'Birth ball', 'Walking'] },
    { category: 'Pain Management', options: ['Natural techniques', 'Epidural', 'Gas and air', 'IV medication'] },
    { category: 'Support', options: ['Partner present', 'Doula', 'Family', 'Midwife'] },
    { category: 'After Birth', options: ['Skin-to-skin', 'Delayed cord clamping', 'Immediate contact', 'Observation'] }
  ]

  return (
    <div className="space-y-12">
      {/* Trimester Selector */}
      <div>
        <h3 className="text-2xl font-bold text-earth mb-6">📅 Trimester Details & Preparation</h3>
        
        <div className="flex gap-3 mb-8 flex-wrap">
          {trimesters.map((trimester) => (
            <button
              key={trimester.id}
              onClick={() => setSelectedTrimester(trimester.id)}
              className={`px-6 py-3 rounded-lg font-bold transition ${
                selectedTrimester === trimester.id
                  ? 'bg-earth text-white shadow-lg'
                  : 'bg-white border-2 border-sand hover:border-earth text-earth'
              }`}
            >
              {trimester.name}
            </button>
          ))}
        </div>

        {/* Selected Trimester Details */}
        <div className={`${trimesters[selectedTrimester].color} border-4 ${trimesters[selectedTrimester].borderColor} rounded-xl p-8`}>
          <div className="grid md:grid-cols-2 gap-8">
            {/* Left side */}
            <div>
              <h4 className="text-2xl font-bold text-earth mb-2">{trimesters[selectedTrimester].name}</h4>
              <p className="text-lg text-orange-600 font-semibold mb-6">{trimesters[selectedTrimester].range}</p>
              
              <div>
                <h5 className="font-bold text-earth mb-3 flex items-center gap-2">
                  <span>✨</span> Focus: {trimesters[selectedTrimester].focus}
                </h5>
                <ul className="space-y-2 mb-6">
                  {trimesters[selectedTrimester].highlights.map((highlight, idx) => (
                    <li key={idx} className="flex gap-2 text-gray-700">
                      <span className="text-green-600">✓</span>
                      <span>{highlight}</span>
                    </li>
                  ))}
                </ul>
              </div>

              <div>
                <h5 className="font-bold text-earth mb-3">📋 Medical Checkups</h5>
                <ul className="space-y-2">
                  {trimesters[selectedTrimester].checkups.map((checkup, idx) => (
                    <li key={idx} className="flex gap-2 text-gray-700">
                      <span className="text-blue-600">📅</span>
                      <span>{checkup}</span>
                    </li>
                  ))}
                </ul>
              </div>
            </div>

            {/* Right side */}
            <div>
              <div className="mb-6">
                <h5 className="font-bold text-earth mb-3 flex items-center gap-2">
                  <span>😊</span> Common Symptoms
                </h5>
                <div className="bg-white/60 rounded p-4">
                  {trimesters[selectedTrimester].symptoms.map((symptom, idx) => (
                    <div key={idx} className="flex gap-2 mb-2">
                      <span className="text-orange-600">•</span>
                      <span className="text-gray-700">{symptom}</span>
                    </div>
                  ))}
                </div>
              </div>

              <div>
                <h5 className="font-bold text-earth mb-3 flex items-center gap-2">
                  <span>🌿</span> Self-Care Tips
                </h5>
                <div className="bg-white/60 rounded p-4">
                  {trimesters[selectedTrimester].selfCare.map((care, idx) => (
                    <div key={idx} className="flex gap-2 mb-2">
                      <span className="text-green-600">→</span>
                      <span className="text-gray-700">{care}</span>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Nutrition Guide */}
      <div>
        <h3 className="text-2xl font-bold text-earth mb-6">🥗 Essential Pregnancy Nutrition</h3>
        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-4">
          {nutritionGuide.map((item, idx) => (
            <div key={idx} className="bg-white rounded-lg border border-sand/30 p-6 hover:shadow-lg transition">
              <h4 className="text-lg font-bold text-earth mb-2">{item.nutrient}</h4>
              <p className="text-sm text-orange-600 font-semibold mb-3">{item.amount}</p>
              
              <div className="mb-3">
                <p className="text-xs font-bold text-gray-600 mb-1">Sources:</p>
                <div className="space-y-1">
                  {item.sources.map((source, i) => (
                    <span key={i} className="text-xs bg-sand/20 text-earth px-2 py-1 rounded inline-block mr-1 mb-1">
                      {source}
                    </span>
                  ))}
                </div>
              </div>
              
              <p className="text-xs text-gray-700 italic border-t border-sand/30 pt-3">
                💡 {item.importance}
              </p>
            </div>
          ))}
        </div>
      </div>

      {/* Birth Plan Template */}
      <div>
        <h3 className="text-2xl font-bold text-earth mb-6">🎯 Create Your Birth Plan</h3>
        <div className="bg-gradient-to-br from-sand/10 to-warm/10 rounded-lg p-8 border border-sand/30">
          <p className="text-gray-700 mb-6">A birth plan helps communicate your preferences to your healthcare team. Consider these important areas:</p>
          
          <div className="grid md:grid-cols-2 gap-6 mb-6">
            {birthPlanTemplate.map((item, idx) => (
              <div key={idx} className="bg-white rounded-lg p-4 border border-sand/30">
                <h5 className="font-bold text-earth mb-3">{item.category}</h5>
                <div className="space-y-2">
                  {item.options.map((option, optIdx) => (
                    <label key={optIdx} className="flex items-center gap-2 cursor-pointer">
                      <input type="checkbox" className="w-4 h-4 text-earth rounded" />
                      <span className="text-sm text-gray-700">{option}</span>
                    </label>
                  ))}
                </div>
              </div>
            ))}
          </div>
          
          <div className="bg-white rounded-lg p-4 border border-sand/30">
            <label className="block">
              <span className="font-bold text-earth mb-2 block">Additional Notes & Preferences:</span>
              <textarea 
                className="w-full px-3 py-2 border border-sand/30 rounded-lg focus:outline-none focus:border-earth" 
                rows={4}
                placeholder="Write any specific requests or concerns..."
              ></textarea>
            </label>
          </div>
          
          <button className="mt-6 bg-earth hover:bg-orange-700 text-white font-bold py-2 px-8 rounded-lg transition">
            Download Birth Plan
          </button>
        </div>
      </div>

      {/* Weekly Checklist */}
      <div>
        <h3 className="text-2xl font-bold text-earth mb-6">✅ Weekly Pregnancy Checklist</h3>
        <div className="grid md:grid-cols-3 gap-6">
          {weeklyChecklist.map((checklist, idx) => (
            <div key={idx} className="bg-white rounded-lg border border-sand/30 p-6">
              <h4 className="text-lg font-bold text-earth mb-4">{checklist.trimester} Trimester</h4>
              <ul className="space-y-3">
                {checklist.items.map((item, itemIdx) => (
                  <label key={itemIdx} className="flex items-start gap-3 cursor-pointer group">
                    <input type="checkbox" className="w-5 h-5 mt-0.5 text-earth rounded cursor-pointer" />
                    <span className="text-gray-700 group-hover:text-earth transition">{item}</span>
                  </label>
                ))}
              </ul>
            </div>
          ))}
        </div>
      </div>

      {/* FAQ Section */}
      <div>
        <h3 className="text-2xl font-bold text-earth mb-6">❓ Frequently Asked Questions</h3>
        <div className="space-y-3">
          {pregnancyFAQ.map((faq) => (
            <div 
              key={faq.id}
              className="bg-white rounded-lg border border-sand/30 overflow-hidden"
            >
              <button
                onClick={() => setExpandedFAQ(expandedFAQ === faq.id ? null : faq.id)}
                className="w-full px-6 py-4 flex items-center justify-between hover:bg-sand/10 transition"
              >
                <span className="font-bold text-earth text-left">{faq.question}</span>
                <span className={`text-2xl transition transform ${expandedFAQ === faq.id ? 'rotate-180' : ''}`}>
                  ▼
                </span>
              </button>
              
              {expandedFAQ === faq.id && (
                <div className="px-6 py-4 bg-sand/5 border-t border-sand/30">
                  <p className="text-gray-700 leading-relaxed">{faq.answer}</p>
                </div>
              )}
            </div>
          ))}
        </div>
      </div>

      {/* Doctor's Information */}
      <div className="bg-gradient-to-r from-forest/10 to-earth/10 rounded-lg p-8 border border-forest/20">
        <h3 className="text-2xl font-bold text-earth mb-6">📞 Important Healthcare Contacts</h3>
        
        <div className="grid md:grid-cols-2 gap-6">
          <div className="bg-white rounded-lg p-6">
            <h5 className="font-bold text-earth mb-4 flex items-center gap-2">
              <span>👨‍⚕️</span> Your Healthcare Providers
            </h5>
            <div className="space-y-3 text-sm text-gray-700">
              <div>
                <p className="font-semibold text-earth">OB/GYN Doctor:</p>
                <input type="text" placeholder="Name & contact" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
              <div>
                <p className="font-semibold text-earth">Midwife:</p>
                <input type="text" placeholder="Name & contact" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
              <div>
                <p className="font-semibold text-earth">Pediatrician:</p>
                <input type="text" placeholder="Name & contact" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
            </div>
          </div>

          <div className="bg-white rounded-lg p-6">
            <h5 className="font-bold text-earth mb-4 flex items-center gap-2">
              <span>🏥</span> Emergency Contacts
            </h5>
            <div className="space-y-3 text-sm text-gray-700">
              <div>
                <p className="font-semibold text-earth">Hospital/Birthing Center:</p>
                <input type="text" placeholder="Name & emergency number" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
              <div>
                <p className="font-semibold text-earth">Emergency Line:</p>
                <input type="text" placeholder="24/7 emergency contact" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
              <div>
                <p className="font-semibold text-earth">Poison Control:</p>
                <input type="text" placeholder="Local poison control" className="w-full mt-1 px-3 py-2 border border-sand/30 rounded" />
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
