'use client'

import { useState } from 'react'

export default function SymptomTracker() {
  const [symptoms, setSymptoms] = useState<{[key: string]: number}>({})
  const [selectedSymptom, setSelectedSymptom] = useState('')
  const [history, setHistory] = useState([
    { date: 'Today', symptom: 'Morning sickness', severity: '7/10', duration: '1 hour' },
    { date: 'Yesterday', symptom: 'Fatigue', severity: '6/10', duration: 'All day' },
    { date: '2 days ago', symptom: 'Lower back pain', severity: '5/10', duration: '2 hours' }
  ])

  const commonSymptoms = [
    { name: 'Morning Sickness/Nausea', trimester: '1st', severity: 9, management: 'Ginger tea, small meals, rest' },
    { name: 'Fatigue', trimester: '1st-3rd', severity: 9, management: 'Rest, sleep, hydration' },
    { name: 'Breast Tenderness', trimester: '1st-2nd', severity: 7, management: 'Supportive bra, warm compress' },
    { name: 'Frequent Urination', trimester: '1st-3rd', severity: 8, management: 'Normal, stay hydrated, Kegels' },
    { name: 'Mood Swings', trimester: '1st-3rd', severity: 8, management: 'Support, exercise, counseling' },
    { name: 'Bloating', trimester: '1st', severity: 6, management: 'Small meals, ginger, movement' },
    { name: 'Lower Back Pain', trimester: '2nd-3rd', severity: 8, management: 'Massage, stretches, supportive shoes' },
    { name: 'Leg Cramps', trimester: '2nd-3rd', severity: 7, management: 'Stretching, magnesium, hydration' },
    { name: 'Swelling (Edema)', trimester: '2nd-3rd', severity: 7, management: 'Elevate feet, compression, rest' },
    { name: 'Shortness of Breath', trimester: '3rd', severity: 6, management: 'Rest, upright position, fresh air' },
    { name: 'Braxton Hicks', trimester: '3rd', severity: 7, management: 'Rest, hydration, timing practice' },
    { name: 'Heartburn', trimester: '1st-3rd', severity: 7, management: 'Small meals, avoid triggers, antacids' },
    { name: 'Constipation', trimester: '1st-3rd', severity: 6, management: 'Fiber, water, exercise, fruits' },
    { name: 'Headaches', trimester: '1st-3rd', severity: 6, management: 'Rest, hydration, massage, cool compress' },
    { name: 'Dizziness', trimester: '2nd-3rd', severity: 5, management: 'Sit down, hydrate, eat regularly' }
  ]

  const logSymptom = (symptom: string, severity: number) => {
    const newEntry = {
      date: new Date().toLocaleDateString(),
      symptom,
      severity: `${severity}/10`,
      duration: 'Not specified'
    }
    setHistory([newEntry, ...history])
    setSelectedSymptom('')
  }

  const ManagementTips = {
    'Morning Sickness/Nausea': ['Drink ginger tea throughout the day', 'Eat small, frequent meals', 'Rest when possible', 'Avoid strong smells', 'Try acupressure wristbands'],
    'Fatigue': ['Get 8-10 hours of sleep', 'Take short naps if possible', 'Eat iron-rich foods', 'Stay hydrated', 'Gentle exercise'],
    'Lower Back Pain': ['Prenatal massage', 'Stretching exercises', 'Wear supportive shoes', 'Use pregnancy pillow', 'Apply warm compress'],
    'Leg Cramps': ['Stretch before bed', 'Stay hydrated', 'Take magnesium supplement (consult doctor)', 'Gentle massage', 'Warm compress']
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-hotpink to-hotpink-dark rounded-2xl p-8 text-white">
        <h1 className="text-4xl font-bold mb-2">Symptom Tracker</h1>
        <p className="text-lg opacity-90">Monitor your pregnancy symptoms and get personalized advice 📋</p>
      </div>

      {/* Quick Log */}
      <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
        <h2 className="text-2xl font-bold text-hotpink mb-6">Log a Symptom</h2>
        
        <div className="grid md:grid-cols-2 gap-6">
          <div>
            <label className="block font-bold text-gray-800 mb-3">Select Symptom</label>
            <div className="space-y-2 max-h-96 overflow-y-auto">
              {commonSymptoms.map((symptom) => (
                <button
                  key={symptom.name}
                  onClick={() => setSelectedSymptom(symptom.name)}
                  className={`w-full text-left p-3 rounded-lg font-semibold transition ${
                    selectedSymptom === symptom.name
                      ? 'bg-hotpink text-white'
                      : 'bg-hotpink/10 text-hotpink hover:bg-hotpink/20'
                  }`}
                >
                  {symptom.name}
                </button>
              ))}
            </div>
          </div>

          <div>
            {selectedSymptom && (
              <div>
                <h3 className="text-xl font-bold text-deepbrown mb-4">{selectedSymptom}</h3>
                
                {/* Severity Selector */}
                <div className="mb-6">
                  <p className="font-bold text-gray-800 mb-3">How severe? (1-10)</p>
                  <div className="grid grid-cols-5 gap-2">
                    {[1, 2, 3, 4, 5, 6, 7, 8, 9, 10].map((num) => (
                      <button
                        key={num}
                        onClick={() => logSymptom(selectedSymptom, num)}
                        className={`p-3 rounded-lg font-bold transition ${
                          num <= 3 ? 'hover:bg-green-200' : num <= 6 ? 'hover:bg-yellow-200' : 'hover:bg-red-200'
                        } bg-gray-100 text-gray-700`}
                      >
                        {num}
                      </button>
                    ))}
                  </div>
                </div>

                {/* Tips for Selected Symptom */}
                <div className="bg-terra/10 rounded-lg p-4 border-l-4 border-terra">
                  <p className="font-bold text-terra mb-2">Management Tips:</p>
                  <ul className="space-y-1 text-sm text-gray-700">
                    {(ManagementTips[selectedSymptom as keyof typeof ManagementTips] || 
                      commonSymptoms.find(s => s.name === selectedSymptom)?.management?.split(', ') || 
                      ['Consult your healthcare provider']).map((tip, idx) => (
                      <li key={idx}>✓ {tip}</li>
                    ))}
                  </ul>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Symptom History */}
      <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
        <h2 className="text-2xl font-bold text-hotpink mb-6">Your Symptom History</h2>
        
        {history.length > 0 ? (
          <div className="space-y-3">
            {history.map((entry, idx) => (
              <div key={idx} className="flex items-center justify-between p-4 bg-hotpink/5 rounded-lg border border-hotpink-light/50">
                <div>
                  <p className="font-bold text-deepbrown">{entry.symptom}</p>
                  <p className="text-sm text-gray-600">{entry.date} • Duration: {entry.duration}</p>
                </div>
                <div className="text-right">
                  <div className={`text-2xl font-bold ${
                    parseInt(entry.severity) <= 3 ? 'text-green-600' :
                    parseInt(entry.severity) <= 6 ? 'text-yellow-600' :
                    'text-red-600'
                  }`}>
                    {entry.severity}
                  </div>
                  <button className="text-xs text-hotpink font-bold mt-1 hover:underline">Delete</button>
                </div>
              </div>
            ))}
          </div>
        ) : (
          <p className="text-center text-gray-600 py-8">No symptoms logged yet. Start tracking when you experience symptoms.</p>
        )}
      </div>

      {/* Symptom Overview */}
      <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
        <h2 className="text-2xl font-bold text-hotpink mb-6">Common Pregnancy Symptoms by Trimester</h2>
        
        <div className="grid md:grid-cols-3 gap-6">
          {[
            {
              trimester: 'First Trimester (Weeks 1-13)',
              symptoms: ['Morning sickness', 'Fatigue', 'Breast tenderness', 'Mood swings', 'Frequent urination'],
              focus: 'Adjusting to pregnancy'
            },
            {
              trimester: 'Second Trimester (Weeks 14-26)',
              symptoms: ['Energy returns', 'Leg cramps', 'Back pain begins', 'Braxton Hicks (later)', 'Increased appetite'],
              focus: 'Growing and showing'
            },
            {
              trimester: 'Third Trimester (Weeks 27-40)',
              symptoms: ['Back pain', 'Swelling', 'Shortness of breath', 'Braxton Hicks', 'Sleeping difficulties'],
              focus: 'Labor preparation'
            }
          ].map((tri, idx) => (
            <div key={idx} className="bg-gradient-to-br from-hotpink/10 to-hotpink-light/10 rounded-lg p-6 border border-hotpink-light">
              <h3 className="font-bold text-deepbrown mb-2">{tri.trimester}</h3>
              <p className="text-sm text-hotpink font-semibold mb-3">Focus: {tri.focus}</p>
              <ul className="space-y-1">
                {tri.symptoms.map((symptom, i) => (
                  <li key={i} className="text-sm text-gray-700">• {symptom}</li>
                ))}
              </ul>
            </div>
          ))}
        </div>
      </div>

      {/* When to Contact Doctor */}
      <div className="bg-red-50 border-2 border-red-300 rounded-xl p-8">
        <h3 className="text-2xl font-bold text-red-900 mb-4 flex items-center gap-2">
          <span>🚨</span> When to Contact Your Healthcare Provider
        </h3>
        
        <div className="space-y-3">
          <div className="flex gap-3">
            <span className="text-2xl">🔴</span>
            <div>
              <p className="font-bold text-red-900">Severe or Unusual Pain</p>
              <p className="text-sm text-red-800">Sharp, severe pain in abdomen, chest, or persistent headaches</p>
            </div>
          </div>
          
          <div className="flex gap-3">
            <span className="text-2xl">🔴</span>
            <div>
              <p className="font-bold text-red-900">Vaginal Bleeding</p>
              <p className="text-sm text-red-800">Any amount of vaginal bleeding</p>
            </div>
          </div>
          
          <div className="flex gap-3">
            <span className="text-2xl">🔴</span>
            <div>
              <p className="font-bold text-red-900">Fever or Chills</p>
              <p className="text-sm text-red-800">Temperature over 100.4°F (38°C)</p>
            </div>
          </div>
          
          <div className="flex gap-3">
            <span className="text-2xl">🔴</span>
            <div>
              <p className="font-bold text-red-900">Severe Swelling</p>
              <p className="text-sm text-red-800">Sudden swelling in face, hands, or feet with headache</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
