'use client'

import { useState, useEffect } from 'react'

export default function Tools() {
  const [activeTab, setActiveTab] = useState('duedate')
  
  // Due Date Calculator
  const [lmpDate, setLmpDate] = useState('')
  const [calculatedDueDate, setCalculatedDueDate] = useState('')
  const [weeksPregnant, setWeeksPregnant] = useState(0)

  const calculateDueDate = () => {
    if (!lmpDate) return
    const date = new Date(lmpDate)
    date.setDate(date.getDate() + 280) // 40 weeks = 280 days
    setCalculatedDueDate(date.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' }))
    
    // Calculate weeks pregnant
    const today = new Date()
    const startDate = new Date(lmpDate)
    const diffTime = Math.abs(today.getTime() - startDate.getTime())
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24))
    setWeeksPregnant(Math.floor(diffDays / 7))
  }

  // Kick Counter
  const [kickCount, setKickCount] = useState(0)
  const [kickTimer, setKickTimer] = useState(false)
  const [kickStartTime, setKickStartTime] = useState<Date | null>(null)
  const [kickDuration, setKickDuration] = useState('')
  const [kickHistory, setKickHistory] = useState<{count: number, duration: string, date: string}[]>([])

  const startKickCounter = () => {
    setKickTimer(true)
    setKickStartTime(new Date())
    setKickCount(0)
  }

  const stopKickCounter = () => {
    setKickTimer(false)
    if (kickStartTime) {
      const endTime = new Date()
      const durationMinutes = Math.floor((endTime.getTime() - kickStartTime.getTime()) / 60000)
      setKickDuration(`${durationMinutes} min`)
      setKickHistory([...kickHistory, { count: kickCount, duration: `${durationMinutes} min`, date: new Date().toLocaleTimeString() }])
    }
  }

  // Contraction Timer
  const [contractionMode, setContractionMode] = useState('inactive') // inactive, timing, recording
  const [contractionStart, setContractionStart] = useState<Date | null>(null)
  const [contractionDuration, setContractionDuration] = useState(0)
  const [lastContractionTime, setLastContractionTime] = useState<Date | null>(null)
  const [timeBetweenContractions, setTimeBetweenContractions] = useState('')
  const [contractionHistory, setContractionHistory] = useState<{duration: number, interval: string, time: string}[]>([])
  const [contractionTimer, setContractionTimer] = useState(0)

  useEffect(() => {
    let interval: NodeJS.Timeout
    if (contractionMode === 'timing' && contractionStart) {
      interval = setInterval(() => {
        const now = new Date()
        const durationSeconds = Math.floor((now.getTime() - contractionStart.getTime()) / 1000)
        setContractionDuration(durationSeconds)
        setContractionTimer(durationSeconds)
      }, 100)
    }
    return () => clearInterval(interval)
  }, [contractionMode, contractionStart])

  const startContraction = () => {
    setContractionMode('timing')
    setContractionStart(new Date())
    setContractionDuration(0)
    setContractionTimer(0)
  }

  const endContraction = () => {
    if (lastContractionTime) {
      const now = new Date()
      const intervalSeconds = Math.floor((now.getTime() - lastContractionTime.getTime()) / 1000)
      const minutes = Math.floor(intervalSeconds / 60)
      const seconds = intervalSeconds % 60
      setTimeBetweenContractions(`${minutes}m ${seconds}s`)
    }
    
    setContractionMode('recording')
    const currentTime = new Date().toLocaleTimeString()
    setContractionHistory([...contractionHistory, {
      duration: contractionDuration,
      interval: timeBetweenContractions || 'First contraction',
      time: currentTime
    }])
    setLastContractionTime(new Date())
  }

  const resetContractionTimer = () => {
    setContractionMode('inactive')
    setContractionStart(null)
    setContractionDuration(0)
    setContractionTimer(0)
    setTimeBetweenContractions('')
    setLastContractionTime(null)
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-hotpink to-hotpink-dark rounded-2xl p-8 text-white">
        <h1 className="text-4xl font-bold mb-2">Pregnancy Tools</h1>
        <p className="text-lg opacity-90">Calculate due dates, count kicks, and time contractions 🛠️</p>
      </div>

      {/* Tool Tabs */}
      <div className="flex gap-3 flex-wrap">
        {[
          { id: 'duedate', label: '📅 Due Date Calculator', emoji: '📅' },
          { id: 'kickcounter', label: '👶 Kick Counter', emoji: '👶' },
          { id: 'contraction', label: '⏱️ Contraction Timer', emoji: '⏱️' }
        ].map((tab) => (
          <button
            key={tab.id}
            onClick={() => setActiveTab(tab.id)}
            className={`px-6 py-3 rounded-lg font-bold transition ${
              activeTab === tab.id
                ? 'bg-hotpink text-white shadow-lg'
                : 'bg-white text-hotpink border-2 border-hotpink-light hover:bg-hotpink/10'
            }`}
          >
            {tab.label}
          </button>
        ))}
      </div>

      {/* Due Date Calculator */}
      {activeTab === 'duedate' && (
        <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
          <h2 className="text-2xl font-bold text-hotpink mb-6">Due Date Calculator</h2>
          
          <div className="grid md:grid-cols-2 gap-8">
            <div>
              <label className="block font-bold text-gray-800 mb-3">Last Menstrual Period (LMP)</label>
              <input
                type="date"
                value={lmpDate}
                onChange={(e) => setLmpDate(e.target.value)}
                className="w-full px-4 py-3 border-2 border-hotpink-light rounded-lg focus:outline-none focus:border-hotpink"
              />
              
              <button
                onClick={calculateDueDate}
                className="w-full mt-4 py-3 bg-hotpink text-white font-bold rounded-lg hover:bg-hotpink-dark transition"
              >
                Calculate Due Date
              </button>

              <div className="mt-6 p-4 bg-terra/10 rounded-lg border-l-4 border-terra">
                <p className="text-sm text-gray-700 mb-2 font-semibold">💡 How to use:</p>
                <p className="text-sm text-gray-600">Select the first day of your last menstrual period. Your due date is calculated as 280 days (40 weeks) from this date.</p>
              </div>
            </div>

            <div>
              {calculatedDueDate && (
                <div className="space-y-4">
                  <div className="bg-gradient-to-br from-hotpink/20 to-hotpink-light/20 rounded-lg p-6 border-2 border-hotpink">
                    <p className="text-sm text-gray-600 mb-1">Your Estimated Due Date</p>
                    <p className="text-4xl font-bold text-hotpink">{calculatedDueDate}</p>
                  </div>

                  <div className="bg-gradient-to-br from-gold/20 to-terra/20 rounded-lg p-6 border-2 border-gold">
                    <p className="text-sm text-gray-600 mb-1">Weeks Pregnant (as of today)</p>
                    <p className="text-4xl font-bold text-terra">{weeksPregnant} weeks</p>
                  </div>

                  <div className="bg-gradient-to-br from-terracotta/20 to-deepbrown/20 rounded-lg p-6 border-2 border-terracotta">
                    <p className="text-sm text-gray-600 mb-1">Trimester</p>
                    <p className="text-4xl font-bold text-deepbrown">
                      {weeksPregnant <= 13 ? '1st' : weeksPregnant <= 26 ? '2nd' : '3rd'}
                    </p>
                  </div>
                </div>
              )}
              
              {!calculatedDueDate && (
                <div className="h-full flex items-center justify-center bg-hotpink/5 rounded-lg border-2 border-dashed border-hotpink-light p-8 text-center">
                  <p className="text-gray-600">Enter your LMP date to calculate your due date</p>
                </div>
              )}
            </div>
          </div>

          {/* Trimester Info */}
          <div className="mt-8 bg-hotpink/5 rounded-xl p-6">
            <h3 className="font-bold text-deepbrown mb-4">Trimester Breakdown</h3>
            <div className="grid md:grid-cols-3 gap-4 text-sm">
              <div className="p-3 bg-white rounded border-l-4 border-terra">
                <p className="font-bold text-terra">1st Trimester</p>
                <p className="text-gray-600">Weeks 1-13</p>
                <p className="text-xs text-gray-500 mt-1">Foundation forming</p>
              </div>
              <div className="p-3 bg-white rounded border-l-4 border-gold">
                <p className="font-bold text-gold">2nd Trimester</p>
                <p className="text-gray-600">Weeks 14-26</p>
                <p className="text-xs text-gray-500 mt-1">Rapid growth</p>
              </div>
              <div className="p-3 bg-white rounded border-l-4 border-terracotta">
                <p className="font-bold text-terracotta">3rd Trimester</p>
                <p className="text-gray-600">Weeks 27-40</p>
                <p className="text-xs text-gray-500 mt-1">Final preparations</p>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Kick Counter */}
      {activeTab === 'kickcounter' && (
        <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
          <h2 className="text-2xl font-bold text-hotpink mb-6">Kick Counter</h2>
          
          <div className="grid md:grid-cols-2 gap-8">
            <div className="flex flex-col items-center justify-center">
              <div className="bg-gradient-to-br from-hotpink/20 to-hotpink-light/20 rounded-full w-40 h-40 flex items-center justify-center border-4 border-hotpink mb-6">
                <div className="text-6xl font-bold text-hotpink">{kickCount}</div>
              </div>

              <div className="space-y-3 w-full">
                {!kickTimer ? (
                  <button
                    onClick={startKickCounter}
                    className="w-full py-4 bg-hotpink text-white font-bold text-lg rounded-lg hover:bg-hotpink-dark transition"
                  >
                    Start Timing
                  </button>
                ) : (
                  <>
                    <button
                      onClick={() => setKickCount(kickCount + 1)}
                      className="w-full py-4 bg-gold text-deepbrown font-bold text-lg rounded-lg hover:bg-gold/80 transition"
                    >
                      Count a Kick
                    </button>
                    <button
                      onClick={stopKickCounter}
                      className="w-full py-4 bg-terracotta text-white font-bold rounded-lg hover:bg-terracotta/80 transition"
                    >
                      Stop Timing
                    </button>
                  </>
                )}
              </div>

              <div className="mt-6 text-center text-gray-600 text-sm">
                <p>🕐 Time: {kickDuration || 'Not started'}</p>
                <p className="mt-2">✓ Kicks per hour: {kickDuration ? Math.round((kickCount / parseInt(kickDuration)) * 60) : '-'}</p>
              </div>
            </div>

            <div>
              <div className="bg-terra/10 rounded-lg p-4 border-l-4 border-terra mb-6">
                <p className="font-bold text-terra mb-2">💡 Kick Counting Tips:</p>
                <ul className="space-y-2 text-sm text-gray-700">
                  <li>✓ Start counting when baby is most active</li>
                  <li>✓ Aim for 10 kicks in less than 2 hours</li>
                  <li>✓ Best done in the afternoon/evening</li>
                  <li>✓ Stop timing once you reach 10 kicks</li>
                  <li>✓ Contact doctor if significantly fewer kicks</li>
                </ul>
              </div>

              <h3 className="font-bold text-deepbrown mb-3">Today's Sessions</h3>
              <div className="space-y-2 max-h-60 overflow-y-auto">
                {kickHistory.length > 0 ? (
                  kickHistory.map((session, idx) => (
                    <div key={idx} className="bg-hotpink/10 rounded-lg p-3 border border-hotpink-light/50">
                      <div className="flex justify-between items-center">
                        <span className="font-bold text-hotpink">{session.count} kicks</span>
                        <span className="text-sm text-gray-600">{session.duration}</span>
                      </div>
                      <p className="text-xs text-gray-500 mt-1">{session.date}</p>
                    </div>
                  ))
                ) : (
                  <p className="text-center text-gray-600 text-sm py-4">No sessions recorded yet</p>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Contraction Timer */}
      {activeTab === 'contraction' && (
        <div className="bg-white rounded-xl border-2 border-hotpink-light p-8">
          <h2 className="text-2xl font-bold text-hotpink mb-6">Contraction Timer</h2>
          
          <div className="grid md:grid-cols-2 gap-8">
            <div className="flex flex-col items-center justify-center">
              <div className={`rounded-full w-48 h-48 flex items-center justify-center border-4 mb-6 transition ${
                contractionMode === 'timing'
                  ? 'bg-red-100 border-red-500'
                  : 'bg-green-100 border-green-500'
              }`}>
                <div className="text-5xl font-bold text-red-600">
                  {contractionTimer}s
                </div>
              </div>

              <div className="text-center mb-6">
                <p className="text-gray-600 text-sm">
                  {contractionMode === 'inactive' && 'Ready to time'}
                  {contractionMode === 'timing' && 'Timing contraction...'}
                  {contractionMode === 'recording' && 'Contraction recorded'}
                </p>
              </div>

              <div className="space-y-3 w-full">
                {contractionMode === 'inactive' && (
                  <button
                    onClick={startContraction}
                    className="w-full py-4 bg-red-500 text-white font-bold text-lg rounded-lg hover:bg-red-600 transition"
                  >
                    Start Contraction
                  </button>
                )}

                {contractionMode === 'timing' && (
                  <button
                    onClick={endContraction}
                    className="w-full py-4 bg-yellow-500 text-white font-bold text-lg rounded-lg hover:bg-yellow-600 transition"
                  >
                    End Contraction
                  </button>
                )}

                {contractionMode === 'recording' && (
                  <>
                    <button
                      onClick={startContraction}
                      className="w-full py-4 bg-red-500 text-white font-bold rounded-lg hover:bg-red-600 transition"
                    >
                      Start Next Contraction
                    </button>
                    <button
                      onClick={resetContractionTimer}
                      className="w-full py-4 bg-gray-400 text-white font-bold rounded-lg hover:bg-gray-500 transition"
                    >
                      Reset Timer
                    </button>
                  </>
                )}
              </div>

              {timeBetweenContractions && contractionMode === 'recording' && (
                <div className="mt-6 p-4 bg-gold/20 rounded-lg border-2 border-gold text-center">
                  <p className="text-sm text-gray-600">Time Since Last Contraction</p>
                  <p className="text-3xl font-bold text-gold">{timeBetweenContractions}</p>
                </div>
              )}
            </div>

            <div>
              <div className="bg-red-50 rounded-lg p-4 border-l-4 border-red-500 mb-6">
                <p className="font-bold text-red-900 mb-2">⏱️ How to Time Contractions:</p>
                <ul className="space-y-2 text-sm text-red-800">
                  <li>✓ Press "Start Contraction" when you feel a contraction beginning</li>
                  <li>✓ Press "End Contraction" when the contraction stops</li>
                  <li>✓ Timer records both duration and interval between contractions</li>
                  <li>✓ Pattern: Contractions 5-1-1 (every 5 min, 1 min long, for 1 hour) = Call doctor</li>
                  <li>✓ Contact healthcare provider if contractions become regular</li>
                </ul>
              </div>

              <h3 className="font-bold text-deepbrown mb-3">Contraction Log</h3>
              <div className="space-y-2 max-h-60 overflow-y-auto">
                {contractionHistory.length > 0 ? (
                  contractionHistory.map((contraction, idx) => (
                    <div key={idx} className="bg-red-50 rounded-lg p-3 border border-red-300/50">
                      <div className="flex justify-between items-start">
                        <div>
                          <p className="font-bold text-red-600">{contraction.duration}s</p>
                          <p className="text-xs text-gray-600">Interval: {contraction.interval}</p>
                        </div>
                        <p className="text-xs text-gray-500">{contraction.time}</p>
                      </div>
                    </div>
                  ))
                ) : (
                  <p className="text-center text-gray-600 text-sm py-4">No contractions timed yet</p>
                )}
              </div>

              {contractionHistory.length >= 6 && (
                <div className="mt-4 p-3 bg-yellow-50 rounded border border-yellow-300">
                  <p className="text-xs font-bold text-yellow-800">
                    📞 After 6 regular contractions, contact your healthcare provider
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  )
}
